function Get-FileSystemItems {
    <#
    .SYNOPSIS
        Retrieves items from local or network file system
    .DESCRIPTION
        Gets files and folders from file system paths
    .PARAMETER Path
        File system path (C:\, \\server\share\, etc.)
    .OUTPUTS
        Array of items with Name, Path, IsDirectory properties
    #>
    param([string]$Path)

    try {
        if (-not (Test-Path $Path)) {
            Write-Warning "Path not found: $Path"
            return @()
        }

        $items = Get-ChildItem -Path $Path -ErrorAction Stop | ForEach-Object {
            $isDir = $_.PSIsContainer

            # Apply extension filter if enabled and it's a file
            if (-not $isDir -and $script:FS_FilterExtensions) {
                if ($_.Extension -notin $script:FS_AllowedExtensions) {
                    return $null
                }
            }

            @{
                Name = $_.Name
                Path = $_.FullName
                IsDirectory = $isDir
                Size = if (-not $isDir) { $_.Length } else { $null }
                Modified = $_.LastWriteTime
            }
        } | Where-Object { $null -ne $_ }

        return $items

    } catch {
        Write-Warning "Error reading directory ${Path}: $_"
        return @()
    }
}

function Get-FileSystemContent {
    <#
    .SYNOPSIS
        Retrieves file content from file system
    .DESCRIPTION
        Reads file content as string
    .PARAMETER Path
        File path
    .OUTPUTS
        File content as string
    #>
    param([string]$Path)

    try {
        if (-not (Test-Path $Path)) {
            throw "File not found: $Path"
        }

        return Get-Content -Path $Path -Raw -ErrorAction Stop

    } catch {
        throw "Failed to read file ${Path}: $_"
    }
}

function Initialize-FileSystemDataSource {
    <#
    .SYNOPSIS
        Initializes and registers the FileSystem data source
    .DESCRIPTION
        Sets up configuration and registers with TreeBuilder
    .PARAMETER AllowedExtensions
        File extensions to include (e.g., @('.ps1', '.bat'))
    .PARAMETER FilterExtensions
        Whether to filter by extensions
    #>
    param(
        [string[]]$AllowedExtensions = @('.ps1', '.bat', '.cmd', '.exe'),
        [bool]$FilterExtensions = $true
    )

    # Set module-level configuration
    $script:FS_AllowedExtensions = $AllowedExtensions
    $script:FS_FilterExtensions = $FilterExtensions

    # Register with TreeBuilder
    if (Get-Command Register-DataSource -ErrorAction SilentlyContinue) {
        # Match local paths (C:\, D:\, etc.) and UNC paths (\\server\share)
        $pathPattern = '^([A-Z]:\\|\\\\[^\\]+\\)'

        Register-DataSource `
            -Name "FileSystem" `
            -PathPattern $pathPattern `
            -GetItemsFunction ${function:Get-FileSystemItems} `
            -GetContentFunction ${function:Get-FileSystemContent}

        Write-Verbose "FileSystem data source registered"
    } else {
        Write-Warning "TreeBuilder module not loaded - FileSystem data source not registered"
    }
}
