function Get-H5AIItems {
    <#
    .SYNOPSIS
        Retrieves items from an H5AI directory
    .DESCRIPTION
        Parses H5AI HTML to extract files and folders
    .PARAMETER Url
        H5AI directory URL
    .OUTPUTS
        Array of items with Name, Path, IsDirectory properties
    #>
    param([string]$Url)

    try {
        if (-not $Url.EndsWith("/")) {
            $Url = "$Url/"
        }

        # Get HTML content (with caching)
        $htmlContent = Get-H5AIWebContent -Url $Url

        # Get the current directory path from URL to strip it from hrefs
        $uri = [System.Uri]$Url
        $currentPath = [System.Web.HttpUtility]::UrlDecode($uri.AbsolutePath).TrimEnd('/')

        $items = @()
        $regex = [regex]'href="([^"]+)"'
        $matches = $regex.Matches($htmlContent)
        $seenNames = @{}

        foreach ($match in $matches) {
            $href = $match.Groups[1].Value

            # Skip parent directory, current directory, and absolute paths to root
            if ($href -eq "../" -or $href -eq "./" -or $href -eq "/" -or $href -match "^\?") {
                continue
            }

            # Skip non-relative URLs (external links, etc.)
            if ($href -match "^https?://") {
                continue
            }

            # Determine if it's a directory (ends with /)
            $isDirectory = $href.EndsWith("/")

            # URL decode the href
            $decodedHref = [System.Web.HttpUtility]::UrlDecode($href)

            # Remove trailing slash for processing
            $cleanPath = $decodedHref.TrimEnd('/')

            # H5AI may return full paths starting from root - strip the current directory path
            if ($cleanPath.StartsWith($currentPath)) {
                $cleanPath = $cleanPath.Substring($currentPath.Length).TrimStart('/')
            }

            # Split by / and get the immediate child (first segment after stripping current path)
            $pathParts = $cleanPath.Split('/', [System.StringSplitOptions]::RemoveEmptyEntries)

            if ($pathParts.Count -eq 0) {
                continue
            }

            # Take the first segment as the immediate child name
            $itemName = $pathParts[0]

            # Skip if name is empty or just "."
            if ([string]::IsNullOrWhiteSpace($itemName) -or $itemName -eq ".") {
                continue
            }

            # Skip duplicates (track by name)
            if ($seenNames.ContainsKey($itemName.ToLower())) {
                continue
            }
            $seenNames[$itemName.ToLower()] = $true

            # If it's a file, check if extension is allowed (if filtering is enabled)
            if (-not $isDirectory -and $script:H5AI_FilterExtensions) {
                if ($itemName -match '\.([^.]+)$') {
                    $extension = ".$($matches[1])"
                    if ($extension -notin $script:H5AI_AllowedExtensions) {
                        continue
                    }
                } else {
                    # No extension found, skip this file
                    continue
                }
            }

            # Build full URL
            $fullUrl = if ($href.StartsWith("/")) {
                # Absolute path on same domain
                $uri = [System.Uri]$Url
                "$($uri.Scheme)://$($uri.Host)$href"
            } else {
                # Relative path - append to current URL
                $baseUrl = if ($Url.EndsWith("/")) { $Url } else { "$Url/" }
                "$baseUrl$itemName$(if ($isDirectory) {'/'} else {''})"
            }

            $items += @{
                Name = $itemName
                Path = $fullUrl
                IsDirectory = $isDirectory
            }
        }

        return $items

    } catch {
        Write-Warning "Error parsing H5AI directory $Url`: $_"
        return @()
    }
}

function Get-H5AIContent {
    <#
    .SYNOPSIS
        Retrieves file content from H5AI server
    .DESCRIPTION
        Downloads file content as string
    .PARAMETER Url
        File URL
    .OUTPUTS
        File content as string
    #>
    param([string]$Url)

    try {
        return Get-H5AIWebContent -Url $Url
    } catch {
        throw "Failed to retrieve content from ${Url}: $_"
    }
}

function Get-H5AIWebContent {
    <#
    .SYNOPSIS
        HTTP wrapper with caching for H5AI
    .DESCRIPTION
        Makes HTTP requests with optional caching, authentication, and timeout
    .PARAMETER Url
        URL to fetch
    .OUTPUTS
        HTML/content as string
    #>
    param([string]$Url)

    # Check cache first
    if ($script:H5AI_WebCache.ContainsKey($Url)) {
        return $script:H5AI_WebCache[$Url]
    }

    try {
        $request = [System.Net.HttpWebRequest]::Create($Url)
        $request.AllowAutoRedirect = $true
        $request.UserAgent = "$($script:H5AI_UserAgent) on $($env:COMPUTERNAME)"
        $request.Timeout = $script:H5AI_Timeout

        # Add HTTP Basic Authentication if enabled
        if ($script:H5AI_UseAuth -and
            -not [string]::IsNullOrWhiteSpace($script:H5AI_Username)) {
            $credentials = "$($script:H5AI_Username):$($script:H5AI_Password)"
            $encodedCred = [System.Convert]::ToBase64String([System.Text.Encoding]::ASCII.GetBytes($credentials))
            $request.Headers.Add("Authorization", "Basic $encodedCred")
        }

        $response = $request.GetResponse()
        $stream = $response.GetResponseStream()
        $reader = New-Object System.IO.StreamReader($stream)
        $content = $reader.ReadToEnd()

        # Cache the result
        $script:H5AI_WebCache[$Url] = $content

        return $content

    } catch {
        throw "HTTP request failed for ${Url}: $_"
    } finally {
        if ($reader) { $reader.Close() }
        if ($stream) { $stream.Close() }
        if ($response) { $response.Close() }
    }
}

function Initialize-H5AIDataSource {
    <#
    .SYNOPSIS
        Initializes and registers the H5AI data source
    .DESCRIPTION
        Sets up configuration and registers with TreeBuilder
    .PARAMETER UserAgent
        User agent string for HTTP requests
    .PARAMETER Timeout
        HTTP timeout in milliseconds
    .PARAMETER UseAuth
        Enable HTTP Basic Authentication
    .PARAMETER Username
        HTTP auth username
    .PARAMETER Password
        HTTP auth password
    .PARAMETER AllowedExtensions
        File extensions to include (e.g., @('.ps1', '.bat'))
    .PARAMETER FilterExtensions
        Whether to filter by extensions
    #>
    param(
        [string]$UserAgent = "ITM H5AI DataSource",
        [int]$Timeout = 30000,
        [bool]$UseAuth = $false,
        [string]$Username = "",
        [string]$Password = "",
        [string[]]$AllowedExtensions = @('.ps1', '.bat', '.cmd', '.exe'),
        [bool]$FilterExtensions = $true
    )

    # Set module-level configuration
    $script:H5AI_UserAgent = $UserAgent
    $script:H5AI_Timeout = $Timeout
    $script:H5AI_UseAuth = $UseAuth
    $script:H5AI_Username = $Username
    $script:H5AI_Password = $Password
    $script:H5AI_AllowedExtensions = $AllowedExtensions
    $script:H5AI_FilterExtensions = $FilterExtensions
    $script:H5AI_WebCache = @{}

    # Register with TreeBuilder
    if (Get-Command Register-DataSource -ErrorAction SilentlyContinue) {
        Register-DataSource `
            -Name "H5AI" `
            -PathPattern "^https?://" `
            -GetItemsFunction ${function:Get-H5AIItems} `
            -GetContentFunction ${function:Get-H5AIContent}

        Write-Verbose "H5AI data source registered"
    } else {
        Write-Warning "TreeBuilder module not loaded - H5AI data source not registered"
    }
}
