function Get-GlobalConfig {
    <#
    .SYNOPSIS
        Fetches and caches the global configuration file
    .DESCRIPTION
        Loads config.json from the root ITM directory
    .PARAMETER BaseUrl
        The base URL being used (e.g., https://files.derenzyit.com/ITM/)
    .OUTPUTS
        Global configuration object or $null
    #>
    param([string]$BaseUrl)

    # Return cached global config if available
    if ($null -ne $script:globalConfig) {
        return $script:globalConfig
    }

    # Return null if config is disabled
    if ([string]::IsNullOrWhiteSpace($script:Config_ConfigFileName)) {
        $script:globalConfig = $null
        return $null
    }

    try {
        # Build global config URL
        $globalConfigUrl = if ($BaseUrl.EndsWith("/")) {
            "$BaseUrl$($script:Config_ConfigFileName)"
        } else {
            "$BaseUrl/$($script:Config_ConfigFileName)"
        }

        # Fetch config using TreeBuilder's Get-ItemContent
        $jsonContent = Get-ItemContent -Path $globalConfigUrl

        # Parse JSON
        $config = $jsonContent | ConvertFrom-Json

        # Cache the result
        $script:globalConfig = $config

        return $config
    }
    catch {
        # No global config file found - return null (not an error)
        $script:globalConfig = $null
        return $null
    }
}

function Get-DirectoryConfig {
    <#
    .SYNOPSIS
        Fetches and parses directory configuration (presets and WinGet apps)
    .DESCRIPTION
        Loads config.json from directory, merges with global config, caches result
    .PARAMETER DirectoryUrl
        URL of the directory to fetch config for
    .PARAMETER BaseUrl
        Base URL for fetching global config (optional)
    .OUTPUTS
        Merged configuration object or $null
    #>
    param(
        [string]$DirectoryUrl,
        [string]$BaseUrl = $null
    )

    # Return null if config is disabled
    if ([string]::IsNullOrWhiteSpace($script:Config_ConfigFileName)) {
        return $null
    }

    # Check cache first
    if ($script:configCache.ContainsKey($DirectoryUrl)) {
        return $script:configCache[$DirectoryUrl]
    }

    # Extract base URL if not provided
    if ([string]::IsNullOrWhiteSpace($BaseUrl)) {
        $uri = [System.Uri]$DirectoryUrl
        $segments = $uri.Segments | Where-Object { $_ -ne '/' }
        if ($segments.Count -gt 1) {
            $parentSegments = $segments[0..($segments.Count - 2)]
            $BaseUrl = $uri.Scheme + "://" + $uri.Host + "/" + ($parentSegments -join "")
        } else {
            $BaseUrl = $uri.Scheme + "://" + $uri.Host + $uri.AbsolutePath
        }
    }

    try {
        # Get global config
        $globalConfig = Get-GlobalConfig -BaseUrl $BaseUrl

        # Build local config file URL
        $configUrl = if ($DirectoryUrl.EndsWith("/")) {
            "$DirectoryUrl$($script:Config_ConfigFileName)"
        } else {
            "$DirectoryUrl/$($script:Config_ConfigFileName)"
        }

        # Try to fetch local config
        $localConfig = $null
        try {
            $jsonContent = Get-ItemContent -Path $configUrl
            $localConfig = $jsonContent | ConvertFrom-Json
        }
        catch {
            # No local config, that's okay
        }

        # Merge configs
        $mergedServiceVerification = Merge-ServiceVerificationConfig `
            -GlobalConfig $globalConfig `
            -LocalConfig $localConfig

        # Build merged config
        if ($null -eq $localConfig) {
            if ($null -ne $globalConfig) {
                $mergedConfig = $globalConfig
                $mergedConfig | Add-Member -NotePropertyName '_serviceVerification' `
                    -NotePropertyValue $mergedServiceVerification -Force
            } else {
                $mergedConfig = [PSCustomObject]@{
                    _serviceVerification = $mergedServiceVerification
                }
            }
        } else {
            $mergedConfig = $localConfig

            # Merge winget section from global if not in local
            if ($null -eq $localConfig.winget -and
                $null -ne $globalConfig -and
                $null -ne $globalConfig.winget) {
                $mergedConfig | Add-Member -NotePropertyName 'winget' `
                    -NotePropertyValue $globalConfig.winget -Force
            }

            # Add merged service verification
            $mergedConfig | Add-Member -NotePropertyName '_serviceVerification' `
                -NotePropertyValue $mergedServiceVerification -Force
        }

        # Cache the result
        $script:configCache[$DirectoryUrl] = $mergedConfig

        return $mergedConfig
    }
    catch {
        # Error processing configs - return null
        $script:configCache[$DirectoryUrl] = $null
        return $null
    }
}

function Merge-ServiceVerificationConfig {
    <#
    .SYNOPSIS
        Merges global and local service verification configurations
    .DESCRIPTION
        Combines service verification rules. Local config takes precedence.
    .PARAMETER GlobalConfig
        Global configuration object
    .PARAMETER LocalConfig
        Local (directory-specific) configuration object
    .OUTPUTS
        Merged service verification configuration
    #>
    param(
        [object]$GlobalConfig,
        [object]$LocalConfig
    )

    $merged = @{
        enabled = $script:Config_ServiceVerificationEnabled
        maxWaitTime = $script:Config_DefaultServiceWaitTime
        checkInterval = $script:Config_ServiceCheckInterval
        services = @()
    }

    # Start with global service verification settings
    if ($null -ne $GlobalConfig -and $null -ne $GlobalConfig.serviceVerification) {
        if ($null -ne $GlobalConfig.serviceVerification.enabled) {
            $merged.enabled = $GlobalConfig.serviceVerification.enabled
        }
        if ($null -ne $GlobalConfig.serviceVerification.maxWaitTime) {
            $merged.maxWaitTime = $GlobalConfig.serviceVerification.maxWaitTime
        }
        if ($null -ne $GlobalConfig.serviceVerification.checkInterval) {
            $merged.checkInterval = $GlobalConfig.serviceVerification.checkInterval
        }
        if ($null -ne $GlobalConfig.serviceVerification.services) {
            $merged.services += $GlobalConfig.serviceVerification.services
        }
    }

    # Override with local service verification settings
    if ($null -ne $LocalConfig -and $null -ne $LocalConfig.serviceVerification) {
        if ($null -ne $LocalConfig.serviceVerification.enabled) {
            $merged.enabled = $LocalConfig.serviceVerification.enabled
        }
        if ($null -ne $LocalConfig.serviceVerification.maxWaitTime) {
            $merged.maxWaitTime = $LocalConfig.serviceVerification.maxWaitTime
        }
        if ($null -ne $LocalConfig.serviceVerification.checkInterval) {
            $merged.checkInterval = $LocalConfig.serviceVerification.checkInterval
        }
        if ($null -ne $LocalConfig.serviceVerification.services) {
            # Add local services, overriding global ones with matching patterns
            foreach ($localSvc in $LocalConfig.serviceVerification.services) {
                # Remove any global service with the same pattern
                $merged.services = @($merged.services | Where-Object {
                    $_.installerPattern -ne $localSvc.installerPattern
                })
                # Add the local service
                $merged.services += $localSvc
            }
        }
    }

    return $merged
}

function Get-ServiceVerificationInfo {
    <#
    .SYNOPSIS
        Gets service verification info for a file based on merged config
    .DESCRIPTION
        Checks if filename matches any service verification patterns
    .PARAMETER FileName
        Name of the file to check
    .PARAMETER MergedConfig
        Merged service verification configuration
    .OUTPUTS
        Hashtable with Services and WaitTime, or $null if no match
    #>
    param(
        [string]$FileName,
        [object]$MergedConfig
    )

    if ([string]::IsNullOrWhiteSpace($FileName) -or
        $null -eq $MergedConfig -or
        -not $MergedConfig.enabled) {
        return $null
    }

    # Match filename against patterns (case-insensitive)
    foreach ($svcDef in $MergedConfig.services) {
        if (Test-WildcardMatch -String $FileName -Pattern $svcDef.installerPattern) {
            return @{
                Services = @($svcDef.name)
                WaitTime = if ($null -ne $svcDef.waitTime) {
                    $svcDef.waitTime
                } else {
                    $MergedConfig.maxWaitTime
                }
                CheckInterval = $MergedConfig.checkInterval
            }
        }
    }

    return $null
}

function Test-WildcardMatch {
    <#
    .SYNOPSIS
        Tests if a string matches a wildcard pattern
    .DESCRIPTION
        Case-insensitive wildcard matching using regex
    .PARAMETER String
        String to test
    .PARAMETER Pattern
        Wildcard pattern (e.g., "*syncro*")
    .OUTPUTS
        Boolean - $true if matches
    #>
    param(
        [string]$String,
        [string]$Pattern
    )

    if ([string]::IsNullOrWhiteSpace($String) -or
        [string]::IsNullOrWhiteSpace($Pattern)) {
        return $false
    }

    # Convert wildcard to regex
    $escaped = [regex]::Escape($Pattern)
    $regexPattern = "^$($escaped -replace '\\\*', '.*' -replace '\\\?', '.')$"

    return $String -match $regexPattern
}
