function Show-PresetSelectionPopup {
    <#
    .SYNOPSIS
        Displays a popup menu for selecting a preset with live preview
    .DESCRIPTION
        Shows available presets, applies them live as you navigate, and returns selection
    #>
    param(
        [Parameter(Mandatory=$true)]
        [object]$Config,

        [Parameter(Mandatory=$true)]
        [TreeNode]$DirectoryNode,

        [Parameter(Mandatory=$true)]
        [TreeNode]$Root
    )

    try { [Console]::CursorVisible = $false } catch { }

    # Extract preset names
    $presetNames = @($Config.presets.PSObject.Properties.Name)

    if ($presetNames.Count -eq 0) {
        return $null
    }

    # Backup current selection state
    $selectionBackup = Backup-SelectionState -Root $Root

    $selectedIndex = 0
    $needsRedraw = $true
    $previousIndex = -1

    # Apply initial preset (first one)
    Apply-Preset -DirectoryNode $DirectoryNode -Config $Config -PresetName $presetNames[$selectedIndex] -AdditiveMode $false
    Update-FlattenedList -Root $Root

    while ($true) {
        # If selection changed, apply the new preset live
        if ($previousIndex -ne $selectedIndex) {
            # Apply the currently selected preset
            Apply-Preset -DirectoryNode $DirectoryNode -Config $Config -PresetName $presetNames[$selectedIndex] -AdditiveMode $false
            Update-FlattenedList -Root $Root

            # Redraw the tree behind the popup
            Draw-Tree -Root $Root -FullRedraw $false -DrawHeaderFooter $false

            $previousIndex = $selectedIndex
            $needsRedraw = $true
        }

        if ($needsRedraw) {
            # Build popup content
            $popupLines = @()
            $popupLines += "================================================================="
            $popupLines += "                SELECT A PRESET (LIVE PREVIEW)                   "
            $popupLines += "================================================================="
            $popupLines += ""

            for ($i = 0; $i -lt $presetNames.Count; $i++) {
                $presetName = $presetNames[$i]
                $prefix = if ($i -eq $selectedIndex) { " > " } else { "   " }
                $popupLines += "$prefix$presetName"
            }

            $popupLines += ""
            $popupLines += "-----------------------------------------------------------------"
            $popupLines += ""
            $popupLines += " [Enter] Confirm selection | [A] Add to previous selection"
            $popupLines += " [Esc] Cancel and restore original"

            # Calculate dimensions
            $maxWidth = ($popupLines | Measure-Object -Property Length -Maximum).Maximum + 4
            $popupHeight = $popupLines.Count + 2

            $windowWidth = [Console]::WindowWidth
            $windowHeight = [Console]::WindowHeight

            # Center the popup
            $startX = [Math]::Max(0, [Math]::Floor(($windowWidth - $maxWidth) / 2))
            $startY = [Math]::Max(0, [Math]::Floor(($windowHeight - $popupHeight) / 2))

            # Draw background box
            for ($y = $startY - 1; $y -lt ($startY + $popupHeight + 1); $y++) {
                if ($y -ge 0 -and $y -lt $windowHeight) {
                    [Console]::SetCursorPosition([Math]::Max(0, $startX - 2), $y)
                    Write-Host -NoNewline "$($script:Theme.PopupBg)$(" " * ($maxWidth + 4))$($script:Theme.Reset)"
                }
            }

            # Draw popup content
            $currentY = $startY
            foreach ($line in $popupLines) {
                if ($currentY -ge 0 -and $currentY -lt $windowHeight) {
                    [Console]::SetCursorPosition($startX, $currentY)
                    $paddedLine = " $line".PadRight($maxWidth - 1) + " "

                    # Determine color
                    if ($line -match "^=+$" -or $line -match "SELECT A PRESET") {
                        Write-Host -NoNewline "$($script:Theme.PopupBg)$($script:Theme.PopupTitleFg)$paddedLine$($script:Theme.Reset)"
                    } elseif ($line -match "^   \>") {
                        # Selected item
                        Write-Host -NoNewline "$($script:Theme.PopupBg)$($script:Theme.PopupHighlightFg)$paddedLine$($script:Theme.Reset)"
                    } elseif ($line -match "^   [^ ]") {
                        # Unselected item
                        Write-Host -NoNewline "$($script:Theme.PopupBg)$($script:Theme.PopupTextFg)$paddedLine$($script:Theme.Reset)"
                    } elseif ($line -match "\[Enter\]|\[A\]|\[Esc\]") {
                        # Help text with key hints
                        Write-Host -NoNewline "$($script:Theme.PopupBg)$($script:Theme.PopupHighlightFg)$paddedLine$($script:Theme.Reset)"
                    } else {
                        Write-Host -NoNewline "$($script:Theme.PopupBg)$($script:Theme.PopupMutedFg)$paddedLine$($script:Theme.Reset)"
                    }
                }
                $currentY++
            }

            $needsRedraw = $false
        }

        # Wait for key press
        $key = [Console]::ReadKey($true)

        switch ($key.Key) {
            "UpArrow" {
                if ($selectedIndex -gt 0) {
                    $selectedIndex--
                }
            }
            "DownArrow" {
                if ($selectedIndex -lt ($presetNames.Count - 1)) {
                    $selectedIndex++
                }
            }
            "Enter" {
                # Return preset with replace mode (keep current live preview)
                return @{
                    PresetName = $presetNames[$selectedIndex]
                    AdditiveMode = $false
                    Cancelled = $false
                }
            }
            "A" {
                # Restore original, then apply in additive mode
                Restore-SelectionState -Root $Root -Backup $selectionBackup
                Apply-Preset -DirectoryNode $DirectoryNode -Config $Config -PresetName $presetNames[$selectedIndex] -AdditiveMode $true

                return @{
                    PresetName = $presetNames[$selectedIndex]
                    AdditiveMode = $true
                    Cancelled = $false
                }
            }
            "Escape" {
                # Restore original selection
                Restore-SelectionState -Root $Root -Backup $selectionBackup

                return @{
                    Cancelled = $true
                }
            }
        }
    }
}

function Backup-SelectionState {
    <#
    .SYNOPSIS
        Backs up the current selection state of all nodes
    .DESCRIPTION
        Recursively walks the tree and stores IsSelected state for restoration
    #>
    param(
        [Parameter(Mandatory=$true)]
        [TreeNode]$Root
    )

    $backup = @{}

    function Recurse-Backup {
        param([TreeNode]$Node)

        # Use URL as unique identifier
        $backup[$Node.Url] = $Node.IsSelected

        foreach ($child in $Node.Children) {
            Recurse-Backup -Node $child
        }
    }

    Recurse-Backup -Node $Root
    return $backup
}

function Restore-SelectionState {
    <#
    .SYNOPSIS
        Restores selection state from a backup
    .DESCRIPTION
        Recursively walks the tree and restores IsSelected from backup
    #>
    param(
        [Parameter(Mandatory=$true)]
        [TreeNode]$Root,

        [Parameter(Mandatory=$true)]
        [hashtable]$Backup
    )

    function Recurse-Restore {
        param([TreeNode]$Node)

        if ($Backup.ContainsKey($Node.Url)) {
            $Node.IsSelected = $Backup[$Node.Url]
        }

        foreach ($child in $Node.Children) {
            Recurse-Restore -Node $child
        }
    }

    Recurse-Restore -Node $Root
}

function Clear-AllSelections {
    <#
    .SYNOPSIS
        Deselects all nodes in the tree
    .DESCRIPTION
        Recursively walks the tree and sets IsSelected = $false for all nodes
    #>
    param(
        [Parameter(Mandatory=$true)]
        [TreeNode]$Root
    )

    function Recurse-ClearSelection {
        param([TreeNode]$Node)

        $Node.IsSelected = $false

        foreach ($child in $Node.Children) {
            Recurse-ClearSelection -Node $child
        }
    }

    Recurse-ClearSelection -Node $Root
}

function Apply-Preset {
    <#
    .SYNOPSIS
        Applies a preset selection to a directory node
    .DESCRIPTION
        Reads the preset definition and selects matching files/WinGet apps in the tree
    #>
    param(
        [Parameter(Mandatory=$true)]
        [TreeNode]$DirectoryNode,

        [Parameter(Mandatory=$true)]
        [object]$Config,

        [Parameter(Mandatory=$true)]
        [string]$PresetName,

        [Parameter(Mandatory=$false)]
        [bool]$AdditiveMode = $false
    )

    # Get the preset definition
    $presetDef = $Config.presets.$PresetName

    if ($null -eq $presetDef) {
        return
    }

    # If not additive mode, clear all selections first
    if (-not $AdditiveMode) {
        # Find the root node
        $rootNode = $DirectoryNode
        while ($null -ne $rootNode.Parent) {
            $rootNode = $rootNode.Parent
        }
        Clear-AllSelections -Root $rootNode
    }

    # Ensure directory is expanded and children are loaded
    if (-not $DirectoryNode.IsExpanded) {
        Expand-TreeNode -Node $DirectoryNode
        $DirectoryNode.IsExpanded = $true
    }

    # Convert preset paths to array if it's not already
    $presetPaths = @($presetDef)

    # Select matching files and WinGet items
    foreach ($item in $presetPaths) {
        # Parse item - can be a string (path only) or object (path + arguments)
        $path = $null
        $arguments = $null

        if ($item -is [string]) {
            $path = $item
        } elseif ($item -is [PSCustomObject] -or $item -is [hashtable]) {
            $path = $item.path
            $arguments = $item.arguments
        } else {
            continue
        }

        if ($path -eq "winget") {
            # Select WinGet parent node only
            $targetNode = Find-NodeByRelativePath -Root $DirectoryNode -RelativePath $path
            if ($null -ne $targetNode) {
                $targetNode.IsSelected = $true
                if ($null -ne $arguments) {
                    $targetNode.Arguments = $arguments
                }
            }
        }
        elseif ($path -match '^winget/(.+)$') {
            # Select specific WinGet application
            $packageId = $matches[1]
            $wingetNode = Find-NodeByRelativePath -Root $DirectoryNode -RelativePath "winget"

            if ($null -ne $wingetNode) {
                # Ensure WinGet node is expanded to load children
                if (-not $wingetNode.IsExpanded) {
                    $wingetNode.IsExpanded = $true
                }

                # Find the specific app child
                foreach ($child in $wingetNode.Children) {
                    if ($child.WinGetPackageId -eq $packageId) {
                        $child.IsSelected = $true
                        if ($null -ne $arguments) {
                            $child.Arguments = $arguments
                        }
                        # Auto-select parent WinGet node (dependency)
                        $wingetNode.IsSelected = $true
                        break
                    }
                }
            }
        }
        else {
            # Regular file path
            $targetNode = Find-NodeByRelativePath -Root $DirectoryNode -RelativePath $path

            if ($null -ne $targetNode) {
                if (-not $targetNode.IsDirectory) {
                    $targetNode.IsSelected = $true
                    if ($null -ne $arguments) {
                        $targetNode.Arguments = $arguments
                    }
                }
            }
        }
    }
}

function Find-NodeByRelativePath {
    <#
    .SYNOPSIS
        Finds a node by its relative path from a root node
    .DESCRIPTION
        Recursively searches the tree for a node matching the relative path
    #>
    param(
        [Parameter(Mandatory=$true)]
        [TreeNode]$Root,

        [Parameter(Mandatory=$true)]
        [string]$RelativePath
    )

    # Split path into segments
    $segments = $RelativePath.Split('/', [StringSplitOptions]::RemoveEmptyEntries)

    if ($segments.Count -eq 0) {
        return $null
    }

    # Start search from root
    $currentNode = $Root

    foreach ($segment in $segments) {
        # Ensure current node is expanded
        if ($currentNode.IsDirectory -and $currentNode.Children.Count -eq 0) {
            Expand-TreeNode -Node $currentNode
        }

        # Find child matching this segment
        $found = $false
        foreach ($child in $currentNode.Children) {
            if ($child.Name -eq $segment) {
                $currentNode = $child
                $found = $true
                break
            }
        }

        if (-not $found) {
            # Path not found
            return $null
        }
    }

    return $currentNode
}
