function Add-WinGetNodesToDirectory {
    <#
    .SYNOPSIS
        Injects WinGet virtual nodes into a directory if config enables it
    .DESCRIPTION
        Creates a virtual "WinGet" parent node with application children based on config.json
    .PARAMETER DirectoryNode
        TreeNode directory to add WinGet nodes to
    #>
    param(
        [Parameter(Mandatory)]
        [TreeNode]$DirectoryNode
    )

    # Get config for this directory
    $config = Get-DirectoryConfig -DirectoryUrl $DirectoryNode.Url

    if ($null -eq $config -or $null -eq $config.winget) {
        return
    }

    # Check if WinGet is enabled
    if ($config.winget.enabled -ne $true) {
        return
    }

    # Check if WinGet applications are defined
    if ($null -eq $config.winget.applications -or $config.winget.applications.Count -eq 0) {
        return
    }

    # Create WinGet parent node (virtual directory)
    $wingetNode = [TreeNode]::new(
        "WinGet",
        $DirectoryNode.Url + "winget",
        $true,
        $DirectoryNode,
        $DirectoryNode.Level + 1
    )
    $wingetNode.IsVirtualWinGet = $true
    $wingetNode.WinGetPackageId = $null  # Parent has no package ID

    # Create children for each application
    foreach ($app in $config.winget.applications) {
        $appNode = [TreeNode]::new(
            $app.name,
            $DirectoryNode.Url + "winget/" + $app.id,
            $false,
            $wingetNode,
            $wingetNode.Level + 1
        )
        $appNode.IsVirtualWinGet = $true
        $appNode.WinGetPackageId = $app.id

        # Add arguments if specified
        if ($null -ne $app.arguments) {
            $appNode.Arguments = $app.arguments
        }

        # Add service verification if specified for this WinGet app
        if ($null -ne $app.verifyServices -and $app.verifyServices.Count -gt 0) {
            $appNode.VerifyServices = @($app.verifyServices)
            $appNode.ServiceWaitTime = if ($null -ne $app.serviceWaitTime) {
                $app.serviceWaitTime
            } else {
                0
            }
        }

        $wingetNode.Children += $appNode
    }

    # Insert WinGet node into directory children (sorted alphabetically)
    $allChildren = @($DirectoryNode.Children) + @($wingetNode)
    $DirectoryNode.Children = $allChildren | Sort-Object `
        @{Expression={-not $_.IsDirectory}}, `
        @{Expression={$_.Name}}
}

function Add-ServiceVerificationToNodes {
    <#
    .SYNOPSIS
        Populates service verification info for file nodes based on config
    .DESCRIPTION
        Checks each file node against merged service verification patterns and
        populates VerifyServices and ServiceWaitTime properties
    .PARAMETER DirectoryNode
        TreeNode directory whose children should be checked
    #>
    param(
        [Parameter(Mandatory)]
        [TreeNode]$DirectoryNode
    )

    # Get config for this directory (includes merged global + local service verification)
    $config = Get-DirectoryConfig -DirectoryUrl $DirectoryNode.Url

    if ($null -eq $config -or $null -eq $config._serviceVerification) {
        return
    }

    $mergedServiceVerification = $config._serviceVerification

    if (-not $mergedServiceVerification.enabled) {
        return
    }

    # Process each child file node
    foreach ($child in $DirectoryNode.Children) {
        # Skip directories and virtual WinGet nodes
        if ($child.IsDirectory -or $child.IsVirtualWinGet) {
            continue
        }

        # Get service verification info for this file
        $svcInfo = Get-ServiceVerificationInfo `
            -FileName $child.Name `
            -MergedConfig $mergedServiceVerification

        if ($null -ne $svcInfo) {
            $child.VerifyServices = $svcInfo.Services
            $child.ServiceWaitTime = $svcInfo.WaitTime
        }
    }
}
