function Start-InteractiveMenu {
    <#
    .SYNOPSIS
        Starts the main interactive tree navigation menu
    .PARAMETER Root
        Root TreeNode of the tree to display
    #>
    param([TreeNode]$Root)

    Update-FlattenedList -Root $Root

    $needsRedraw = $true
    $needsFullRedraw = $true
    $lastScrollOffset = $script:scrollOffset

    # Initial draw
    Draw-Tree -Root $Root -FullRedraw $true -DrawHeaderFooter $true

    while ($true) {
        # Update scroll offset to keep current selection visible
        $windowHeight = [Console]::WindowHeight - $script:Config_ReservedLines
        if ($script:currentIndex -lt $script:scrollOffset) {
            $script:scrollOffset = $script:currentIndex
            $needsRedraw = $true
            $needsFullRedraw = $true
        }
        if ($script:currentIndex -ge ($script:scrollOffset + $windowHeight)) {
            $script:scrollOffset = $script:currentIndex - $windowHeight + 1
            $needsRedraw = $true
            $needsFullRedraw = $true
        }

        # Check if scroll offset changed (need full redraw)
        if ($lastScrollOffset -ne $script:scrollOffset) {
            $needsFullRedraw = $true
            $lastScrollOffset = $script:scrollOffset
        }

        # Only redraw if something changed
        if ($needsRedraw) {
            # When list structure changes but no scroll, we can do selective redraw
            # without redrawing header/footer
            Draw-Tree -Root $Root -FullRedraw $needsFullRedraw -DrawHeaderFooter $needsFullRedraw
            $needsRedraw = $false
            $needsFullRedraw = $false
        }

        # Wait for key press (non-blocking with timeout)
        if ([Console]::KeyAvailable) {
            $key = [Console]::ReadKey($true)
            $currentNode = $script:flattenedNodes[$script:currentIndex]

            switch ($key.Key) {
                "UpArrow" {
                    if ($script:currentIndex -gt 0) {
                        $script:currentIndex--
                        $needsRedraw = $true
                    }
                }
                "DownArrow" {
                    if ($script:currentIndex -lt ($script:flattenedNodes.Count - 1)) {
                        $script:currentIndex++
                        $needsRedraw = $true
                    }
                }
                "RightArrow" {
                    if ($currentNode.IsDirectory -and -not $currentNode.IsExpanded) {
                        Expand-TreeNode -Node $currentNode
                        $currentNode.IsExpanded = $true
                        Update-FlattenedList -Root $Root
                        $needsRedraw = $true
                    }
                }
                "LeftArrow" {
                    if ($currentNode.IsDirectory -and $currentNode.IsExpanded) {
                        $currentNode.IsExpanded = $false
                        Update-FlattenedList -Root $Root
                        $needsRedraw = $true
                    }
                }
                "Enter" {
                    Toggle-NodeExpansion -Node $currentNode
                    Update-FlattenedList -Root $Root
                    $needsRedraw = $true
                }
                "Spacebar" {
                    Toggle-NodeSelection -Node $currentNode
                    $needsRedraw = $true
                }
                "R" {
                    try {
                        $selectedNodes = Get-SelectedScripts -Root $Root
                        Invoke-SelectedScripts -SelectedNodes $selectedNodes
                        # Force immediate full redraw after returning
                        Draw-Tree -Root $Root -FullRedraw $true -DrawHeaderFooter $true
                        $needsRedraw = $false
                        $needsFullRedraw = $false
                    } catch {
                        try { [Console]::Clear() } catch { Clear-Host }
                        Write-Host "$($script:Theme.ErrorFg)`nError: $_$($script:Theme.Reset)"
                        Write-Host "$($script:Theme.InfoFg)`nPress any key to return to menu...$($script:Theme.Reset)"
                        $null = [Console]::ReadKey($true)
                        Draw-Tree -Root $Root -FullRedraw $true -DrawHeaderFooter $true
                        $needsRedraw = $false
                        $needsFullRedraw = $false
                    }
                }
                "P" {
                    # Preset key - only works on directories
                    if ($currentNode.IsDirectory) {
                        try {
                            # Fetch config for this directory
                            $config = Get-DirectoryConfig -DirectoryUrl $currentNode.Url

                            if ($null -ne $config -and $null -ne $config.presets) {
                                # Show preset selection popup with live preview
                                $presetResult = Show-PresetSelectionPopup -Config $config `
                                                                           -DirectoryNode $currentNode `
                                                                           -Root $Root

                                # If not cancelled, the preset is already applied (live preview)
                                # If it was additive mode, it was re-applied with additive flag
                                # Update the flattened list either way
                                if ($null -ne $presetResult) {
                                    Update-FlattenedList -Root $Root
                                }
                            }

                            # Redraw the tree
                            Draw-Tree -Root $Root -FullRedraw $true -DrawHeaderFooter $true
                            $needsRedraw = $false
                            $needsFullRedraw = $false
                        } catch {
                            # Silently ignore errors - no config file is not an error
                            Draw-Tree -Root $Root -FullRedraw $true -DrawHeaderFooter $true
                            $needsRedraw = $false
                            $needsFullRedraw = $false
                        }
                    }
                }
                "Escape" {
                    return
                }
            }
        }

        Start-Sleep -Milliseconds $script:Config_PollingInterval
    }
}
