function Toggle-NodeExpansion {
    <#
    .SYNOPSIS
        Toggles the expansion state of a directory node
    .PARAMETER Node
        TreeNode to toggle
    #>
    param([TreeNode]$Node)

    if ($Node.IsDirectory) {
        if (-not $Node.IsExpanded) {
            # Expanding - lazy load children if needed
            Expand-TreeNode -Node $Node
        }
        $Node.IsExpanded = -not $Node.IsExpanded
    }
}

function Toggle-NodeSelection {
    <#
    .SYNOPSIS
        Toggles the selection state of a node
    .PARAMETER Node
        TreeNode to toggle
    #>
    param([TreeNode]$Node)

    # Don't allow selection of the root directory
    if ($null -eq $Node.Parent) {
        return
    }

    # Don't allow selection of empty directories
    if ($Node.IsDirectory -and $Node.Children.Count -eq 0 -and $Node.IsExpanded -and -not $Node.IsVirtualWinGet) {
        return
    }

    $Node.IsSelected = -not $Node.IsSelected

    # Special handling for WinGet child nodes
    if ($Node.IsVirtualWinGet -and $null -ne $Node.WinGetPackageId -and $Node.IsSelected) {
        # Child WinGet app selected - auto-select parent WinGet node
        if ($null -ne $Node.Parent -and $Node.Parent.IsVirtualWinGet) {
            $Node.Parent.IsSelected = $true
        }
    }

    # If deselecting WinGet parent, deselect all children
    if ($Node.IsVirtualWinGet -and $null -eq $Node.WinGetPackageId -and -not $Node.IsSelected) {
        foreach ($child in $Node.Children) {
            $child.IsSelected = $false
        }
    }

    # If it's a folder, toggle all children
    if ($Node.IsDirectory -and $Node.Children.Count -gt 0 -and -not $Node.IsVirtualWinGet) {
        Set-ChildrenSelection -Node $Node -Selected $Node.IsSelected
    }
}

function Set-ChildrenSelection {
    <#
    .SYNOPSIS
        Recursively sets selection state of all children
    .PARAMETER Node
        Parent TreeNode
    .PARAMETER Selected
        Selection state to apply
    #>
    param(
        [TreeNode]$Node,
        [bool]$Selected
    )

    # Expand node to load children if not already loaded
    if ($Node.IsDirectory -and $Node.Children.Count -eq 0) {
        Expand-TreeNode -Node $Node
    }

    foreach ($child in $Node.Children) {
        if (-not $child.IsDirectory) {
            $child.IsSelected = $Selected
        }
        if ($child.IsDirectory -and $child.Children.Count -gt 0) {
            Set-ChildrenSelection -Node $child -Selected $Selected
        }
    }
}

function Get-SelectedScripts {
    <#
    .SYNOPSIS
        Collects all selected nodes from the tree
    .PARAMETER Root
        Root TreeNode to search from
    .OUTPUTS
        Array of selected TreeNode objects
    #>
    param([TreeNode]$Root)

    $script:selectedNodes = [System.Collections.ArrayList]::new()

    function Recurse-Selected {
        param([TreeNode]$Node)

        if (-not $Node.IsDirectory -and $Node.IsSelected) {
            [void]$script:selectedNodes.Add($Node)
        }

        foreach ($child in $Node.Children) {
            Recurse-Selected -Node $child
        }
    }

    Recurse-Selected -Node $Root
    return $script:selectedNodes.ToArray()
}
