function New-ServiceVerificationScript {
    <#
    .SYNOPSIS
        Generates PowerShell code for service verification
    .DESCRIPTION
        Creates script code to wait for services to be installed and running.
        Based on logic from Syncro Legacy script.
    .PARAMETER ServiceNames
        Array of service names to verify
    .PARAMETER MaxWaitTime
        Maximum time (seconds) to wait for services
    .PARAMETER CheckInterval
        Polling interval (seconds) when checking services
    .PARAMETER InstallerName
        Name of the installer (for display purposes)
    .OUTPUTS
        String containing PowerShell code to inject into execution script
    #>
    param(
        [string[]]$ServiceNames,
        [int]$MaxWaitTime,
        [int]$CheckInterval,
        [string]$InstallerName
    )

    if ($null -eq $ServiceNames -or $ServiceNames.Count -eq 0) {
        return ""
    }

    $serviceList = ($ServiceNames | ForEach-Object { "'$_'" }) -join ', '

    $script = @"
    # Service Verification for: $InstallerName
    `$servicesToVerify = @($serviceList)
    `$serviceWaitTime = $MaxWaitTime
    `$serviceCheckInterval = $CheckInterval

    if (`$servicesToVerify.Count -gt 0) {
        Write-Host ""
        Write-Host " Waiting for services to be installed..." -ForegroundColor Yellow

        # Phase 1: Wait for services to exist
        `$startTime = Get-Date
        `$allInstalled = `$false

        while (-not `$allInstalled -and ((Get-Date) - `$startTime).TotalSeconds -lt `$serviceWaitTime) {
            `$installedCount = 0
            foreach (`$svcName in `$servicesToVerify) {
                if (Get-Service -Name `$svcName -ErrorAction SilentlyContinue) {
                    `$installedCount++
                }
            }

            if (`$installedCount -eq `$servicesToVerify.Count) {
                `$allInstalled = `$true
                Write-Host ""
                Write-Host " [OK] " -NoNewline -ForegroundColor Green
                Write-Host "All services installed successfully!" -ForegroundColor Green
            } else {
                Start-Sleep -Seconds `$serviceCheckInterval
                Write-Host "." -NoNewline -ForegroundColor Yellow
            }
        }

        if (-not `$allInstalled) {
            Write-Host ""
            Write-Host " [WARN] " -NoNewline -ForegroundColor Yellow
            Write-Host "Not all services were installed within the timeout period." -ForegroundColor Yellow
            `$serviceVerificationFailed = `$true
        }

        # Phase 2: Wait for services to be running (only if all were installed)
        if (`$allInstalled) {
            Write-Host " Waiting for services to start..." -ForegroundColor Yellow
            `$allRunning = `$false
            `$startTime = Get-Date

            while (-not `$allRunning -and ((Get-Date) - `$startTime).TotalSeconds -lt `$serviceWaitTime) {
                `$runningCount = 0

                foreach (`$svcName in `$servicesToVerify) {
                    `$service = Get-Service -Name `$svcName -ErrorAction SilentlyContinue
                    if (`$service -and `$service.Status -eq 'Running') {
                        `$runningCount++
                    }
                }

                if (`$runningCount -eq `$servicesToVerify.Count) {
                    `$allRunning = `$true
                    Write-Host ""
                    Write-Host " [OK] " -NoNewline -ForegroundColor Green
                    Write-Host "All services are running!" -ForegroundColor Green
                } else {
                    Start-Sleep -Seconds `$serviceCheckInterval
                    Write-Host "." -NoNewline -ForegroundColor Yellow
                }
            }

            if (-not `$allRunning) {
                Write-Host ""
                Write-Host " [WARN] " -NoNewline -ForegroundColor Yellow
                Write-Host "Not all services started within the timeout period." -ForegroundColor Yellow
                `$serviceVerificationFailed = `$true
            }
        }

        # Display service status
        Write-Host ""
        Write-Host " Service Status:" -ForegroundColor Cyan
        foreach (`$svcName in `$servicesToVerify) {
            `$service = Get-Service -Name `$svcName -ErrorAction SilentlyContinue
            if (`$service) {
                `$statusColor = if (`$service.Status -eq 'Running') { 'Green' } else { 'Yellow' }
                Write-Host "  - `${svcName}: " -NoNewline
                Write-Host "`$(`$service.Status)" -ForegroundColor `$statusColor
            } else {
                Write-Host "  - `${svcName}: " -NoNewline
                Write-Host "Not Found" -ForegroundColor Red
            }
        }
        Write-Host ""
    }
"@

    return $script
}

function New-ExecutionScript {
    <#
    .SYNOPSIS
        Creates a temporary PowerShell script that downloads and executes selected installers
    .PARAMETER SelectedNodes
        Array of TreeNode objects representing selected files/WinGet apps
    .OUTPUTS
        String - Path to the temporary execution script
    #>
    param([TreeNode[]]$SelectedNodes)

    # Create download directory if it doesn't exist
    if (-not (Test-Path $script:Config_DownloadPath)) {
        New-Item -ItemType Directory -Path $script:Config_DownloadPath -Force | Out-Null
    }

    # Separate WinGet nodes from regular file nodes
    $wingetNodes = $SelectedNodes | Where-Object { $_.IsVirtualWinGet }
    $regularNodes = $SelectedNodes | Where-Object { -not $_.IsVirtualWinGet }

    # Separate WinGet parent from WinGet apps
    $wingetParent = $wingetNodes | Where-Object { $null -eq $_.WinGetPackageId }
    $wingetApps = $wingetNodes | Where-Object { $null -ne $_.WinGetPackageId }

    # Calculate total items (WinGet apps + regular files)
    $totalItems = $wingetApps.Count + $regularNodes.Count

    # Create a temporary script that will download and run all selected files
    $tempScript = [System.IO.Path]::GetTempFileName()
    $tempScript = [System.IO.Path]::ChangeExtension($tempScript, '.ps1')

    $scriptContent = @'
# Installer Execution Script (Web-Based)
$Host.UI.RawUI.WindowTitle = 'Installing Selected Software'
Write-Host ""
Write-Host "=================================================================" -ForegroundColor Cyan
Write-Host "           INSTALLATION IN PROGRESS" -ForegroundColor Cyan
Write-Host "=================================================================" -ForegroundColor Cyan
Write-Host ""
'@ + "`nWrite-Host `" Total items to install: $totalItems`" -ForegroundColor White`n" + @'
Write-Host ""

$installerNumber = 0
$downloadPath =
'@ + "`"$($script:Config_DownloadPath)`"`n" + @'

# Result tracking
$successfulInstalls = @()
$failedInstalls = @()
$skippedInstalls = @()
$wingetInstalled = $false
$wingetInstallFailed = $false

# Service verification tracking
$serviceVerificationFailed = $false
$allServicesToVerify = @()
'@ + "`n"

    # If WinGet parent is selected OR any WinGet apps are selected, check/install WinGet
    if ($wingetParent -or $wingetApps.Count -gt 0) {
        $scriptContent += "`n"
        $scriptContent += "# Check if WinGet is installed`n"
        $scriptContent += "Write-Host `"-----------------------------------------------------------------`" -ForegroundColor Gray`n"
        $scriptContent += "Write-Host `"`"`n"
        $scriptContent += "Write-Host `" Checking for WinGet installation...`" -ForegroundColor White`n"
        $scriptContent += "Write-Host `"`"`n"
        $scriptContent += "if (-not (Get-Command winget -ErrorAction SilentlyContinue)) {`n"
        $scriptContent += "    Write-Host `" [!] `" -NoNewline -ForegroundColor Yellow`n"
        $scriptContent += "    Write-Host `"WinGet not found. Installing WinGet...`" -ForegroundColor Yellow`n"
        $scriptContent += "    Write-Host `"`"`n"
        $scriptContent += "    try {`n"
        $scriptContent += "        # Run winget.pro installer in isolated process to prevent exit from closing terminal`n"
        $scriptContent += "        powershell.exe -NoProfile -Command `"irm winget.pro | iex`" | Out-Null`n"
        $scriptContent += "        Write-Host `"`"`n"
        $scriptContent += "        Write-Host `" [OK] `" -NoNewline -ForegroundColor Green`n"
        $scriptContent += "        Write-Host `"WinGet installed successfully`" -ForegroundColor Green`n"
        $scriptContent += "        `$wingetInstalled = `$true`n"
        $scriptContent += "    } catch {`n"
        $scriptContent += "        Write-Host `"`"`n"
        $scriptContent += "        Write-Host `" [ERROR] `" -NoNewline -ForegroundColor Red`n"
        $scriptContent += "        Write-Host `"Failed to install WinGet: `$_`" -ForegroundColor Red`n"
        $scriptContent += "        Write-Host `"`"`n"
        $scriptContent += "        Write-Host `" Skipping WinGet applications...`" -ForegroundColor Yellow`n"
        $scriptContent += "        `$wingetInstallFailed = `$true`n"
        $scriptContent += "    }`n"
        $scriptContent += "} else {`n"
        $scriptContent += "    Write-Host `" [OK] `" -NoNewline -ForegroundColor Green`n"
        $scriptContent += "    Write-Host `"WinGet is already installed`" -ForegroundColor Green`n"
        $scriptContent += "    `$wingetInstalled = `$true`n"
        $scriptContent += "}`n"
        $scriptContent += "Write-Host `"`"`n"
    }

    # Add WinGet app installation commands
    foreach ($app in $wingetApps) {
        $appName = $app.Name
        $packageId = $app.WinGetPackageId
        $customArgs = $app.Arguments

        $scriptContent += "`n`n"
        $scriptContent += "`$installerNumber++`n"
        $scriptContent += "Write-Host `"-----------------------------------------------------------------`" -ForegroundColor Gray`n"
        $scriptContent += "Write-Host `"`"`n"
        $scriptContent += "Write-Host `" [`$installerNumber/$totalItems] `" -NoNewline -ForegroundColor Yellow`n"
        $scriptContent += "Write-Host `"Installing via WinGet: `" -NoNewline -ForegroundColor White`n"
        $scriptContent += "Write-Host `"$appName`" -ForegroundColor Cyan`n"
        $scriptContent += "Write-Host `" Package ID: $packageId`" -ForegroundColor Gray`n"

        if ($null -ne $customArgs -and $customArgs.Trim() -ne "") {
            $scriptContent += "Write-Host `" Arguments: $customArgs`" -ForegroundColor Gray`n"
        }

        $scriptContent += "Write-Host `"`"`n"

        # Build argument list for WinGet
        $wingetArgList = "'install', '$packageId', '--accept-package-agreements', '--accept-source-agreements'"
        if ($null -ne $customArgs -and $customArgs.Trim() -ne "") {
            # Add custom arguments to WinGet command
            $escapedArgs = $customArgs -replace "'", "''"
            $wingetArgList += ", '$escapedArgs'"
        }

        $scriptContent += "if (Get-Command winget -ErrorAction SilentlyContinue) {`n"
        $scriptContent += "    try {`n"
        $scriptContent += "        `$process = Start-Process -FilePath 'winget' -ArgumentList $wingetArgList -Wait -PassThru -NoNewWindow`n"
        $scriptContent += "        `$exitCode = `$process.ExitCode`n"
        $scriptContent += "        if (`$exitCode -eq 0) {`n"
        $scriptContent += "            Write-Host `"`"`n"
        $scriptContent += "            Write-Host `" [OK] `" -NoNewline -ForegroundColor Green`n"
        $scriptContent += "            Write-Host `"$appName installed successfully`" -ForegroundColor Green`n"
        $scriptContent += "            `$successfulInstalls += [PSCustomObject]@{Name='$appName'; Type='WinGet'; ExitCode=`$exitCode}`n"

        # Add service verification for WinGet app if configured
        if ($null -ne $app.VerifyServices -and $app.VerifyServices.Count -gt 0) {
            $waitTime = if ($app.ServiceWaitTime -gt 0) { $app.ServiceWaitTime } else { $script:Config_DefaultServiceWaitTime }
            $checkInterval = $script:Config_ServiceCheckInterval
            $svcScript = New-ServiceVerificationScript -ServiceNames $app.VerifyServices -MaxWaitTime $waitTime -CheckInterval $checkInterval -InstallerName $appName
            $scriptContent += $svcScript
            # Track services for potential re-check
            foreach ($svc in $app.VerifyServices) {
                $scriptContent += "            `$allServicesToVerify += '$svc'`n"
            }
        }

        $scriptContent += "        } else {`n"
        $scriptContent += "            Write-Host `"`"`n"
        $scriptContent += "            Write-Host `" [FAIL] `" -NoNewline -ForegroundColor Red`n"
        $scriptContent += "            Write-Host `"$appName installation exited with code: `$exitCode`" -ForegroundColor Red`n"
        $scriptContent += "            `$failedInstalls += [PSCustomObject]@{Name='$appName'; Type='WinGet'; ExitCode=`$exitCode; Error='Exit code: ' + `$exitCode}`n"
        $scriptContent += "        }`n"
        $scriptContent += "    } catch {`n"
        $scriptContent += "        Write-Host `"`"`n"
        $scriptContent += "        Write-Host `" [ERROR] `" -NoNewline -ForegroundColor Red`n"
        $scriptContent += "        Write-Host `"Error installing $appName`: `$_`" -ForegroundColor Red`n"
        $scriptContent += "        `$failedInstalls += [PSCustomObject]@{Name='$appName'; Type='WinGet'; ExitCode=-1; Error=`$_.Exception.Message}`n"
        $scriptContent += "    }`n"
        $scriptContent += "} else {`n"
        $scriptContent += "    Write-Host `" [SKIP] `" -NoNewline -ForegroundColor Yellow`n"
        $scriptContent += "    Write-Host `"WinGet not available, skipping $appName`" -ForegroundColor Yellow`n"
        $scriptContent += "    `$skippedInstalls += [PSCustomObject]@{Name='$appName'; Type='WinGet'; Reason='WinGet not available'}`n"
        $scriptContent += "}`n"
        $scriptContent += "Write-Host `"`"`n"
    }

    # Add regular file download and execution commands
    foreach ($node in $regularNodes) {
        $fileName = $node.Name
        $fileUrl = $node.Url
        $localPath = Join-Path $script:Config_DownloadPath $fileName
        $ext = [System.IO.Path]::GetExtension($fileName).ToLower()
        $customArgs = $node.Arguments

        $scriptContent += "`n`n"
        $scriptContent += "`$installerNumber++`n"
        $scriptContent += "Write-Host `"-----------------------------------------------------------------`" -ForegroundColor Gray`n"
        $scriptContent += "Write-Host `"`"`n"
        $scriptContent += "Write-Host `" [`$installerNumber/$totalItems] `" -NoNewline -ForegroundColor Yellow`n"
        $scriptContent += "Write-Host `"Downloading: `" -NoNewline -ForegroundColor White`n"
        $scriptContent += "Write-Host `"$fileName`" -ForegroundColor Cyan`n"

        if ($null -ne $customArgs -and $customArgs.Trim() -ne "") {
            $scriptContent += "Write-Host `" Arguments: $customArgs`" -ForegroundColor Gray`n"
        }

        $scriptContent += "Write-Host `"`"`n"

        # Download section
        $escapedUrl = $fileUrl -replace "'", "''"
        $escapedPath = $localPath -replace "'", "''"

        $scriptContent += "try {`n"
        $scriptContent += "    `$webClient = New-Object System.Net.WebClient`n"

        # Add authentication to download if enabled
        if ($script:Config_UseHttpAuth -and -not [string]::IsNullOrWhiteSpace($script:Config_HttpAuthUsername)) {
            $escapedUsername = $script:Config_HttpAuthUsername -replace "'", "''"
            $escapedPassword = $script:Config_HttpAuthPassword -replace "'", "''"
            $scriptContent += "    # Add HTTP Basic Authentication`n"
            $scriptContent += "    `$credentials = '$escapedUsername' + ':' + '$escapedPassword'`n"
            $scriptContent += "    `$encodedCredentials = [System.Convert]::ToBase64String([System.Text.Encoding]::ASCII.GetBytes(`$credentials))`n"
            $scriptContent += "    `$webClient.Headers.Add('Authorization', 'Basic ' + `$encodedCredentials)`n"
        }

        $scriptContent += "    `$webClient.DownloadFile('$escapedUrl', '$escapedPath')`n"
        $scriptContent += "    `$webClient.Dispose()`n"
        $scriptContent += "    Write-Host `" [OK] `" -NoNewline -ForegroundColor Green`n"
        $scriptContent += "    Write-Host `"Download completed`" -ForegroundColor Green`n"
        $scriptContent += "    Write-Host `"`"`n"
        $scriptContent += "    Write-Host `" Launching: `" -NoNewline -ForegroundColor White`n"
        $scriptContent += "    Write-Host `"$fileName`" -ForegroundColor Cyan`n"
        $scriptContent += "    Write-Host `"`"`n`n"

        # Execution section
        $escapedArgs = if ($null -ne $customArgs) { $customArgs -replace "'", "''" } else { "" }

        switch ($ext) {
            ".ps1" {
                $scriptContent += "    # Execute PowerShell script with bypass`n"
                if ($null -ne $customArgs -and $customArgs.Trim() -ne "") {
                    $scriptContent += "    `$process = Start-Process -FilePath 'powershell.exe' -ArgumentList '-ExecutionPolicy', 'Bypass', '-NoProfile', '-File', '$escapedPath', '$escapedArgs' -Wait -PassThru -NoNewWindow`n"
                } else {
                    $scriptContent += "    `$process = Start-Process -FilePath 'powershell.exe' -ArgumentList '-ExecutionPolicy', 'Bypass', '-NoProfile', '-File', '$escapedPath' -Wait -PassThru -NoNewWindow`n"
                }
                $scriptContent += "    `$exitCode = `$process.ExitCode`n"
            }
            {$_ -in ".bat", ".cmd"} {
                $scriptContent += "    # Execute batch/cmd file`n"
                if ($null -ne $customArgs -and $customArgs.Trim() -ne "") {
                    $scriptContent += "    `$process = Start-Process -FilePath 'cmd.exe' -ArgumentList '/c', '$escapedPath', '$escapedArgs' -Wait -PassThru -NoNewWindow`n"
                } else {
                    $scriptContent += "    `$process = Start-Process -FilePath 'cmd.exe' -ArgumentList '/c', '$escapedPath' -Wait -PassThru -NoNewWindow`n"
                }
                $scriptContent += "    `$exitCode = `$process.ExitCode`n"
            }
            ".exe" {
                $scriptContent += "    # Execute installer`n"
                if ($null -ne $customArgs -and $customArgs.Trim() -ne "") {
                    $scriptContent += "    `$process = Start-Process -FilePath '$escapedPath' -ArgumentList '$escapedArgs' -Wait -PassThru`n"
                } else {
                    $scriptContent += "    `$process = Start-Process -FilePath '$escapedPath' -Wait -PassThru`n"
                }
                $scriptContent += "    `$exitCode = `$process.ExitCode`n"
            }
            default {
                $scriptContent += "    Write-Host `" [ERROR] Unsupported file type: $ext`" -ForegroundColor Red`n"
                $scriptContent += "    `$exitCode = -1`n"
            }
        }

        $scriptContent += "`n    if (`$exitCode -eq 0) {`n"
        $scriptContent += "        Write-Host `"`"`n"
        $scriptContent += "        Write-Host `" [OK] `" -NoNewline -ForegroundColor Green`n"
        $scriptContent += "        Write-Host `"$fileName completed successfully`" -ForegroundColor Green`n"
        $scriptContent += "        `$successfulInstalls += [PSCustomObject]@{Name='$fileName'; Type='File'; ExitCode=`$exitCode}`n"

        # Add service verification for regular file if configured
        if ($null -ne $node.VerifyServices -and $node.VerifyServices.Count -gt 0) {
            $waitTime = if ($node.ServiceWaitTime -gt 0) { $node.ServiceWaitTime } else { $script:Config_DefaultServiceWaitTime }
            $checkInterval = $script:Config_ServiceCheckInterval
            $svcScript = New-ServiceVerificationScript -ServiceNames $node.VerifyServices -MaxWaitTime $waitTime -CheckInterval $checkInterval -InstallerName $fileName
            $scriptContent += $svcScript
            # Track services for potential re-check
            foreach ($svc in $node.VerifyServices) {
                $scriptContent += "        `$allServicesToVerify += '$svc'`n"
            }
        }

        $scriptContent += "    } else {`n"
        $scriptContent += "        Write-Host `"`"`n"
        $scriptContent += "        Write-Host `" [FAIL] `" -NoNewline -ForegroundColor Red`n"
        $scriptContent += "        Write-Host `"$fileName exited with code: `$exitCode`" -ForegroundColor Red`n"
        $scriptContent += "        `$failedInstalls += [PSCustomObject]@{Name='$fileName'; Type='File'; ExitCode=`$exitCode; Error='Exit code: ' + `$exitCode}`n"
        $scriptContent += "    }`n"
        $scriptContent += "} catch {`n"
        $scriptContent += "    Write-Host `"`"`n"
        $scriptContent += "    Write-Host `" [ERROR] `" -NoNewline -ForegroundColor Red`n"
        $scriptContent += "    Write-Host `"Error processing $fileName`: `$_`" -ForegroundColor Red`n"
        $scriptContent += "    `$failedInstalls += [PSCustomObject]@{Name='$fileName'; Type='File'; ExitCode=-1; Error=`$_.Exception.Message}`n"
        $scriptContent += "}`n"
        $scriptContent += "Write-Host `"`"`n"
    }

    $escapedTempScript = $tempScript -replace "'", "''"
    $scriptContent += "`n"

    # Add final service re-check if any verification failed
    $scriptContent += "# Re-check all services if any verification failed`n"
    $scriptContent += "if (`$serviceVerificationFailed -and `$allServicesToVerify.Count -gt 0) {`n"
    $scriptContent += "    Write-Host `"=================================================================`" -ForegroundColor Cyan`n"
    $scriptContent += "    Write-Host `"`"`n"
    $scriptContent += "    Write-Host `" RE-CHECKING SERVICE STATUS`" -ForegroundColor Cyan`n"
    $scriptContent += "    Write-Host `"`"`n"
    $scriptContent += "    Write-Host `" Some services failed to start initially. Re-checking now...`" -ForegroundColor Yellow`n"
    $scriptContent += "    Write-Host `"`"`n"
    $scriptContent += "    `$uniqueServices = `$allServicesToVerify | Select-Object -Unique`n"
    $scriptContent += "    foreach (`$svcName in `$uniqueServices) {`n"
    $scriptContent += "        `$service = Get-Service -Name `$svcName -ErrorAction SilentlyContinue`n"
    $scriptContent += "        if (`$service) {`n"
    $scriptContent += "            `$statusColor = if (`$service.Status -eq 'Running') { 'Green' } else { 'Yellow' }`n"
    $scriptContent += "            Write-Host `"   `${svcName}: `" -NoNewline -ForegroundColor White`n"
    $scriptContent += "            Write-Host `"`$(`$service.Status)`" -ForegroundColor `$statusColor`n"
    $scriptContent += "        } else {`n"
    $scriptContent += "            Write-Host `"   `${svcName}: `" -NoNewline -ForegroundColor White`n"
    $scriptContent += "            Write-Host `"Not Found`" -ForegroundColor Red`n"
    $scriptContent += "        }`n"
    $scriptContent += "    }`n"
    $scriptContent += "    Write-Host `"`"`n"
    $scriptContent += "}`n"
    $scriptContent += "`n"

    $scriptContent += "Write-Host `"=================================================================`" -ForegroundColor Cyan`n"
    $scriptContent += "Write-Host `"`"`n"
    $scriptContent += "Write-Host `" INSTALLATION SUMMARY`" -ForegroundColor Cyan`n"
    $scriptContent += "Write-Host `"`"`n"
    $scriptContent += "Write-Host `" Total items processed: `$installerNumber`" -ForegroundColor White`n"
    $scriptContent += "Write-Host `"`"`n"
    $scriptContent += "`n"
    $scriptContent += "# Display successful installations`n"
    $scriptContent += "if (`$successfulInstalls.Count -gt 0) {`n"
    $scriptContent += "    Write-Host `" SUCCESS (`$(`$successfulInstalls.Count)) `" -ForegroundColor Black -BackgroundColor Green`n"
    $scriptContent += "    Write-Host `"`"`n"
    $scriptContent += "    foreach (`$item in `$successfulInstalls) {`n"
    $scriptContent += "        Write-Host `"   [OK] `" -NoNewline -ForegroundColor Green`n"
    $scriptContent += "        Write-Host `"`$(`$item.Name)`" -NoNewline -ForegroundColor White`n"
    $scriptContent += "        Write-Host `" (`$(`$item.Type))`" -ForegroundColor Gray`n"
    $scriptContent += "    }`n"
    $scriptContent += "    Write-Host `"`"`n"
    $scriptContent += "}`n"
    $scriptContent += "`n"
    $scriptContent += "# Display failed installations`n"
    $scriptContent += "if (`$failedInstalls.Count -gt 0) {`n"
    $scriptContent += "    Write-Host `" FAILED (`$(`$failedInstalls.Count)) `" -ForegroundColor White -BackgroundColor Red`n"
    $scriptContent += "    Write-Host `"`"`n"
    $scriptContent += "    foreach (`$item in `$failedInstalls) {`n"
    $scriptContent += "        Write-Host `"   [X] `" -NoNewline -ForegroundColor Red`n"
    $scriptContent += "        Write-Host `"`$(`$item.Name)`" -NoNewline -ForegroundColor White`n"
    $scriptContent += "        Write-Host `" (`$(`$item.Type))`" -ForegroundColor Gray`n"
    $scriptContent += "        Write-Host `"       Error: `$(`$item.Error)`" -ForegroundColor Red`n"
    $scriptContent += "    }`n"
    $scriptContent += "    Write-Host `"`"`n"
    $scriptContent += "}`n"
    $scriptContent += "`n"
    $scriptContent += "# Display skipped installations`n"
    $scriptContent += "if (`$skippedInstalls.Count -gt 0) {`n"
    $scriptContent += "    Write-Host `" SKIPPED (`$(`$skippedInstalls.Count)) `" -ForegroundColor Black -BackgroundColor Yellow`n"
    $scriptContent += "    Write-Host `"`"`n"
    $scriptContent += "    foreach (`$item in `$skippedInstalls) {`n"
    $scriptContent += "        Write-Host `"   [-] `" -NoNewline -ForegroundColor Yellow`n"
    $scriptContent += "        Write-Host `"`$(`$item.Name)`" -NoNewline -ForegroundColor White`n"
    $scriptContent += "        Write-Host `" (`$(`$item.Type))`" -ForegroundColor Gray`n"
    $scriptContent += "        Write-Host `"       Reason: `$(`$item.Reason)`" -ForegroundColor Yellow`n"
    $scriptContent += "    }`n"
    $scriptContent += "    Write-Host `"`"`n"
    $scriptContent += "}`n"
    $scriptContent += "`n"
    $scriptContent += "# Display WinGet installation status if attempted`n"
    $scriptContent += "if (`$wingetInstalled) {`n"
    $scriptContent += "    Write-Host `" WinGet Status: `" -NoNewline -ForegroundColor White`n"
    $scriptContent += "    Write-Host `"Installed and ready`" -ForegroundColor Green`n"
    $scriptContent += "    Write-Host `"`"`n"
    $scriptContent += "} elseif (`$wingetInstallFailed) {`n"
    $scriptContent += "    Write-Host `" WinGet Status: `" -NoNewline -ForegroundColor White`n"
    $scriptContent += "    Write-Host `"Installation failed`" -ForegroundColor Red`n"
    $scriptContent += "    Write-Host `"`"`n"
    $scriptContent += "}`n"
    $scriptContent += "`n"
    $scriptContent += "Write-Host `"=================================================================`" -ForegroundColor Cyan`n"
    $scriptContent += "Write-Host `"`"`n"
    $scriptContent += "if (`$failedInstalls.Count -eq 0 -and `$skippedInstalls.Count -eq 0) {`n"
    $scriptContent += "    Write-Host `" All installations completed successfully!`" -ForegroundColor Green`n"
    $scriptContent += "} elseif (`$successfulInstalls.Count -gt 0) {`n"
    $scriptContent += "    Write-Host `" Installation completed with some issues. Please review above.`" -ForegroundColor Yellow`n"
    $scriptContent += "} else {`n"
    $scriptContent += "    Write-Host `" Installation completed with errors. Please review above.`" -ForegroundColor Red`n"
    $scriptContent += "}`n"
    $scriptContent += "Write-Host `"`"`n"
    $scriptContent += "Write-Host `" Press any key to close this window...`" -ForegroundColor Gray`n"
    $scriptContent += "`$null = `$Host.UI.RawUI.ReadKey('NoEcho,IncludeKeyDown')`n`n"

    # Add cleanup code only if configured
    if ($script:Config_CleanupTempScripts) {
        $scriptContent += "# Clean up temp script`n"
        $scriptContent += "Remove-Item '$escapedTempScript' -Force -ErrorAction SilentlyContinue`n"
    } else {
        $scriptContent += "# Temp script cleanup disabled (Config_CleanupTempScripts = `$false)`n"
        $scriptContent += "# Temp script preserved at: $escapedTempScript`n"
    }

    Set-Content -Path $tempScript -Value $scriptContent -Encoding UTF8
    return $tempScript
}
