function Show-LicenseConfirmation {
    <#
    .SYNOPSIS
        Shows a popup confirmation dialog for licensed software
    .PARAMETER LicensedProducts
        Hashtable of licensed products (key=FriendlyName, value=array of filenames)
    .OUTPUTS
        Boolean - $true if user confirms, $false if cancelled
    #>
    param([hashtable]$LicensedProducts)

    try { [Console]::CursorVisible = $false } catch { }

    $productCount = $LicensedProducts.Keys.Count

    # Build popup content
    $popupLines = @()
    $popupLines += "================================================================="
    $popupLines += "                LICENSE CONFIRMATION                             "
    $popupLines += "================================================================="
    $popupLines += ""
    $popupLines += " You have selected $productCount licensed product(s):"
    $popupLines += ""

    foreach ($productName in ($LicensedProducts.Keys | Sort-Object)) {
        $files = $LicensedProducts[$productName]
        $popupLines += "  * $productName"
        foreach ($file in $files) {
            $popupLines += "    - $file"
        }
        $popupLines += ""
    }

    $popupLines += " These products may require licensing or subscription fees."
    $popupLines += " Please ensure proper authorization before installation."
    $popupLines += ""
    $popupLines += "-----------------------------------------------------------------"
    $popupLines += ""
    $popupLines += " Do you want to continue? [Y]es / [N]o: "

    # Calculate dimensions
    $maxWidth = ($popupLines | Measure-Object -Property Length -Maximum).Maximum + 4  # +4 for padding
    $popupHeight = $popupLines.Count + 2  # +2 for spacing

    $windowWidth = [Console]::WindowWidth
    $windowHeight = [Console]::WindowHeight

    # Center the popup
    $startX = [Math]::Max(0, [Math]::Floor(($windowWidth - $maxWidth) / 2))
    $startY = [Math]::Max(0, [Math]::Floor(($windowHeight - $popupHeight) / 2))

    # Draw background box with buffer
    for ($y = $startY - 1; $y -lt ($startY + $popupHeight + 1); $y++) {
        if ($y -ge 0 -and $y -lt $windowHeight) {
            [Console]::SetCursorPosition([Math]::Max(0, $startX - 2), $y)
            Write-Host -NoNewline "$($script:Theme.PopupBg)$(" " * ($maxWidth + 4))$($script:Theme.Reset)"
        }
    }

    # Draw popup content
    $currentY = $startY
    foreach ($line in $popupLines) {
        if ($currentY -ge 0 -and $currentY -lt $windowHeight) {
            [Console]::SetCursorPosition($startX, $currentY)
            $paddedLine = " $line".PadRight($maxWidth - 1) + " "

            # Determine color
            if ($line -match "^=+$" -or $line -match "LICENSE CONFIRMATION") {
                Write-Host -NoNewline "$($script:Theme.PopupBg)$($script:Theme.PopupTitleFg)$paddedLine$($script:Theme.Reset)"
            } elseif ($line -match "licensed product") {
                Write-Host -NoNewline "$($script:Theme.PopupBg)$($script:Theme.PopupTextFg)$paddedLine$($script:Theme.Reset)"
            } elseif ($line -match "^\s+\*") {
                Write-Host -NoNewline "$($script:Theme.PopupBg)$($script:Theme.PopupHighlightFg)$paddedLine$($script:Theme.Reset)"
            } elseif ($line -match "^\s+-") {
                Write-Host -NoNewline "$($script:Theme.PopupBg)$($script:Theme.PopupMutedFg)$paddedLine$($script:Theme.Reset)"
            } elseif ($line -match "require licensing") {
                Write-Host -NoNewline "$($script:Theme.PopupBg)$($script:Theme.PopupWarningFg)$paddedLine$($script:Theme.Reset)"
            } elseif ($line -match "Do you want") {
                # Split the line to color Yes/No differently
                [Console]::SetCursorPosition($startX, $currentY)
                Write-Host -NoNewline "$($script:Theme.PopupBg)$($script:Theme.PopupTextFg) Do you want to continue? $($script:Theme.Reset)"
                Write-Host -NoNewline "$($script:Theme.PopupBg)$($script:Theme.PopupYesFg)[Y]$($script:Theme.Reset)"
                Write-Host -NoNewline "$($script:Theme.PopupBg)$($script:Theme.PopupTextFg)es / $($script:Theme.Reset)"
                Write-Host -NoNewline "$($script:Theme.PopupBg)$($script:Theme.PopupNoFg)[N]$($script:Theme.Reset)"
                Write-Host -NoNewline "$($script:Theme.PopupBg)$($script:Theme.PopupTextFg)o: $(" " * ($maxWidth - 33))$($script:Theme.Reset)"
            } else {
                Write-Host -NoNewline "$($script:Theme.PopupBg)$($script:Theme.PopupMutedFg)$paddedLine$($script:Theme.Reset)"
            }
        }
        $currentY++
    }

    # Wait for Y/N input
    while ($true) {
        $key = [Console]::ReadKey($true)
        if ($key.Key -eq 'Y') {
            return $true
        }
        elseif ($key.Key -eq 'N' -or $key.Key -eq 'Escape') {
            return $false
        }
    }
}
