function Download-File {
    <#
    .SYNOPSIS
        Downloads a file from a URL to a local path
    .PARAMETER Url
        URL to download from
    .PARAMETER DestinationPath
        Local path to save the file
    .OUTPUTS
        Boolean - $true if successful, $false otherwise
    #>
    param(
        [string]$Url,
        [string]$DestinationPath
    )

    try {
        # Ensure destination directory exists
        $destinationDir = [System.IO.Path]::GetDirectoryName($DestinationPath)
        if (-not (Test-Path $destinationDir)) {
            New-Item -ItemType Directory -Path $destinationDir -Force | Out-Null
        }

        $webClient = New-Object System.Net.WebClient
        $webClient.DownloadFile($Url, $DestinationPath)
        $webClient.Dispose()

        return $true
    }
    catch {
        Write-Host "$($script:Theme.ErrorFg)Error downloading file: $_$($script:Theme.Reset)"
        return $false
    }
}

function Test-WindowsTerminal {
    <#
    .SYNOPSIS
        Checks if Windows Terminal is installed
    .OUTPUTS
        Boolean - $true if available, $false otherwise
    #>
    $wtPath = Get-Command wt.exe -ErrorAction SilentlyContinue
    return ($null -ne $wtPath)
}

function Get-LicensedSoftwareInSelection {
    <#
    .SYNOPSIS
        Checks selected nodes for licensed software
    .PARAMETER SelectedNodes
        Array of TreeNode objects to check
    .OUTPUTS
        Hashtable of licensed products found (key=FriendlyName, value=array of filenames)
    #>
    param([TreeNode[]]$SelectedNodes)

    $licensedProducts = @{}

    foreach ($node in $SelectedNodes) {
        $fileName = $node.Name

        foreach ($license in $script:LicensedSoftware) {
            foreach ($keyword in $license.Keywords) {
                if ($fileName -match [regex]::Escape($keyword)) {
                    $friendlyName = $license.FriendlyName

                    # Track which files match this licensed product
                    if (-not $licensedProducts.ContainsKey($friendlyName)) {
                        $licensedProducts[$friendlyName] = @()
                    }
                    $licensedProducts[$friendlyName] += $fileName
                    break  # Don't check other keywords for this license entry
                }
            }
        }
    }

    return $licensedProducts
}

function Invoke-SelectedScripts {
    <#
    .SYNOPSIS
        Executes selected scripts/installers in a new window
    .PARAMETER SelectedNodes
        Array of TreeNode objects to execute
    #>
    param([TreeNode[]]$SelectedNodes)

    if ($SelectedNodes.Count -eq 0) {
        try { [Console]::Clear() } catch { Clear-Host }
        Write-Host "$($script:Theme.WarningFg)`nNo scripts selected!`n$($script:Theme.Reset)"
        Write-Host "$($script:Theme.InfoFg)Press any key to continue...$($script:Theme.Reset)"
        $null = [Console]::ReadKey($true)
        return
    }

    # Check for licensed software
    $licensedProducts = Get-LicensedSoftwareInSelection -SelectedNodes $SelectedNodes

    if ($licensedProducts.Count -gt 0) {
        $confirmed = Show-LicenseConfirmation -LicensedProducts $licensedProducts

        if (-not $confirmed) {
            try { [Console]::Clear() } catch { Clear-Host }
            Write-Host "$($script:Theme.WarningFg)`nInstallation cancelled by user.`n$($script:Theme.Reset)"
            Write-Host "$($script:Theme.InfoFg)Press any key to return to menu...$($script:Theme.Reset)"
            $null = [Console]::ReadKey($true)
            return
        }
    }

    # Create execution script
    $executionScript = New-ExecutionScript -SelectedNodes $SelectedNodes

    # Determine which terminal to use based on -Terminal parameter
    $useWindowsTerminal = $false
    $hasWindowsTerminal = Test-WindowsTerminal
    $inWindowsTerminal = $null -ne $env:WT_SESSION

    switch ($Terminal) {
        'Auto' {
            # If in WT, stay in WT. If WT available, use it. Otherwise, PowerShell.
            $useWindowsTerminal = $hasWindowsTerminal
        }
        'WindowsTerminal' {
            if ($hasWindowsTerminal) {
                $useWindowsTerminal = $true
            } else {
                Write-Host "$($script:Theme.WarningFg)Windows Terminal not found. Using PowerShell console instead.$($script:Theme.Reset)"
                $useWindowsTerminal = $false
            }
        }
        'PowerShell' {
            $useWindowsTerminal = $false
        }
    }

    try {
        if ($useWindowsTerminal) {
            # Launch in Windows Terminal
            # Use '-w 0' to open in same instance if already in Windows Terminal
            Start-Process wt.exe -ArgumentList '-w', '0', 'new-tab', 'powershell.exe', '-ExecutionPolicy', 'Bypass', '-NoProfile', '-File', "`"$executionScript`""
        } else {
            # Fallback to regular PowerShell window
            Start-Process powershell.exe -ArgumentList '-ExecutionPolicy', 'Bypass', '-NoProfile', '-NoExit', '-File', "`"$executionScript`""
        }

        # Show confirmation in current window
        try { [Console]::Clear() } catch { Clear-Host }
        Write-Host ""
        Write-Host "$($script:Theme.SuccessFg)[OK] $($script:Theme.Reset)Installation launched in new window"
        Write-Host ""
        Write-Host "$($script:Theme.InfoFg)  The installers will be downloaded and executed in a separate terminal window.$($script:Theme.Reset)"
        Write-Host "$($script:Theme.InfoFg)  You can continue using this menu or close it.$($script:Theme.Reset)"
        Write-Host ""
        Write-Host "$($script:Theme.InfoFg)Press any key to return to menu...$($script:Theme.Reset)"
        $null = [Console]::ReadKey($true)

    } catch {
        Write-Host "$($script:Theme.ErrorFg)`nError launching installers: $_`n$($script:Theme.Reset)"
        Write-Host "$($script:Theme.InfoFg)Press any key to return to menu...$($script:Theme.Reset)"
        $null = [Console]::ReadKey($true)
    }
}
