function Register-DataSource {
    <#
    .SYNOPSIS
        Registers a data source provider
    .DESCRIPTION
        Adds a data source to the registry. Data sources can handle specific path patterns.
    .PARAMETER Name
        Unique identifier for the data source (e.g., "H5AI", "FileSystem")
    .PARAMETER PathPattern
        Regex pattern to match paths this source handles (e.g., "^https?://", "^[A-Z]:\\")
    .PARAMETER GetItemsFunction
        ScriptBlock that retrieves items: { param($Path) }
    .PARAMETER GetContentFunction
        ScriptBlock that retrieves content: { param($Path) }
    .EXAMPLE
        Register-DataSource -Name "H5AI" -PathPattern "^https?://" `
            -GetItemsFunction ${function:Get-H5AIItems} `
            -GetContentFunction ${function:Get-H5AIContent}
    #>
    param(
        [Parameter(Mandatory)]
        [string]$Name,

        [Parameter(Mandatory)]
        [string]$PathPattern,

        [Parameter(Mandatory)]
        [scriptblock]$GetItemsFunction,

        [Parameter(Mandatory)]
        [scriptblock]$GetContentFunction
    )

    if (-not $script:DataSourceRegistry) {
        $script:DataSourceRegistry = @{}
    }

    $script:DataSourceRegistry[$Name] = @{
        Name = $Name
        PathPattern = $PathPattern
        GetItems = $GetItemsFunction
        GetContent = $GetContentFunction
    }

    Write-Verbose "Registered data source: $Name (pattern: $PathPattern)"
}

function Get-DataSourceForPath {
    <#
    .SYNOPSIS
        Finds the appropriate data source for a given path
    .DESCRIPTION
        Matches path against registered data source patterns
    .PARAMETER Path
        Path to match against data sources
    .OUTPUTS
        Data source object or $null if no match
    #>
    param([string]$Path)

    if (-not $script:DataSourceRegistry) {
        throw "No data sources registered. Import a data source module first."
    }

    foreach ($ds in $script:DataSourceRegistry.Values) {
        if ($Path -match $ds.PathPattern) {
            return $ds
        }
    }

    throw "No data source found for path: $Path"
}

function Build-Tree {
    <#
    .SYNOPSIS
        Builds a TreeNode hierarchy from any data source
    .DESCRIPTION
        Creates initial tree structure by calling the appropriate data source.
        Automatically selects data source based on path pattern.
    .PARAMETER RootPath
        Starting path (URL, file path, UNC path, etc.)
    .PARAMETER RootName
        Display name for root node
    .PARAMETER Parent
        Parent TreeNode (for recursive calls)
    .PARAMETER Level
        Current depth level
    .OUTPUTS
        TreeNode representing the hierarchy
    .EXAMPLE
        # With H5AI data source registered
        $tree = Build-Tree -RootPath "https://files.example.com/ITM/" -RootName "ITM"
    .EXAMPLE
        # With FileSystem data source registered
        $tree = Build-Tree -RootPath "C:\Scripts\" -RootName "Scripts"
    #>
    param(
        [Parameter(Mandatory)]
        [string]$RootPath,

        [Parameter(Mandatory)]
        [string]$RootName,

        [TreeNode]$Parent = $null,

        [int]$Level = 0
    )

    # Create root node
    $node = [TreeNode]::new($RootName, $RootPath, $true, $Parent, $Level)

    # Get appropriate data source for this path
    $dataSource = Get-DataSourceForPath -Path $RootPath

    try {
        # Call data source to get items
        $items = & $dataSource.GetItems $RootPath

        if ($null -eq $items) {
            $items = @()
        }

        # Sort: directories first, then by name
        $sortedItems = $items | Sort-Object @{Expression={-not $_.IsDirectory}}, @{Expression={$_.Name}}

        # Create child nodes (lazy loading - don't fetch grandchildren yet)
        foreach ($item in $sortedItems) {
            $childNode = [TreeNode]::new(
                $item.Name,
                $item.Path,
                $item.IsDirectory,
                $node,
                $Level + 1
            )

            # Store any additional metadata
            if ($item.Size) { $childNode.Size = $item.Size }
            if ($item.Modified) { $childNode.Modified = $item.Modified }

            $node.Children += $childNode
        }

    } catch {
        Write-Warning "Error building tree for ${RootPath}: $_"
        # Return node with no children (will show as empty folder)
    }

    return $node
}

function Expand-TreeNode {
    <#
    .SYNOPSIS
        Lazy-loads children for a TreeNode
    .DESCRIPTION
        Expands a directory node by fetching its children from the appropriate data source.
        Automatically selects data source based on node's path.
    .PARAMETER Node
        TreeNode to expand
    .EXAMPLE
        Expand-TreeNode -Node $folderNode
    #>
    param(
        [Parameter(Mandatory)]
        [TreeNode]$Node
    )

    # Only expand directories
    if (-not $Node.IsDirectory) {
        return
    }

    # Skip if already loaded (has children)
    if ($Node.Children.Count -gt 0) {
        return
    }

    # Get appropriate data source for this path
    $dataSource = Get-DataSourceForPath -Path $Node.Url

    try {
        # Call data source to get items
        $items = & $dataSource.GetItems $Node.Url

        if ($null -eq $items) {
            $items = @()
        }

        # Sort: directories first, then by name
        $sortedItems = $items | Sort-Object @{Expression={-not $_.IsDirectory}}, @{Expression={$_.Name}}

        # Create child nodes
        $Node.Children = @()
        foreach ($item in $sortedItems) {
            $childNode = [TreeNode]::new(
                $item.Name,
                $item.Path,
                $item.IsDirectory,
                $Node,
                $Node.Level + 1
            )

            # Store any additional metadata
            if ($item.Size) { $childNode.Size = $item.Size }
            if ($item.Modified) { $childNode.Modified = $item.Modified }

            $Node.Children += $childNode
        }

    } catch {
        Write-Warning "Error expanding node $($Node.Name): $_"
    }
}

function Get-ItemContent {
    <#
    .SYNOPSIS
        Retrieves content from any data source
    .DESCRIPTION
        Downloads or reads file content using the appropriate data source.
        Automatically selects data source based on path pattern.
    .PARAMETER Path
        Path to the item (URL, file path, etc.)
    .OUTPUTS
        Content as string or bytes
    .EXAMPLE
        $content = Get-ItemContent -Path "https://files.example.com/script.ps1"
    #>
    param(
        [Parameter(Mandatory)]
        [string]$Path
    )

    # Get appropriate data source for this path
    $dataSource = Get-DataSourceForPath -Path $Path

    # Call data source's GetContent function
    return & $dataSource.GetContent $Path
}

function Download-ItemToPath {
    <#
    .SYNOPSIS
        Downloads an item to a local path
    .DESCRIPTION
        Retrieves content from any data source and saves to local file
    .PARAMETER SourcePath
        Source path (URL, UNC, etc.)
    .PARAMETER DestinationPath
        Local file system path to save to
    .OUTPUTS
        Boolean - success or failure
    #>
    param(
        [Parameter(Mandatory)]
        [string]$SourcePath,

        [Parameter(Mandatory)]
        [string]$DestinationPath
    )

    try {
        # Ensure destination directory exists
        $destinationDir = [System.IO.Path]::GetDirectoryName($DestinationPath)
        if (-not (Test-Path $destinationDir)) {
            New-Item -ItemType Directory -Path $destinationDir -Force | Out-Null
        }

        # Get content from data source
        $content = Get-ItemContent -Path $SourcePath

        # Save to file
        if ($content -is [byte[]]) {
            [System.IO.File]::WriteAllBytes($DestinationPath, $content)
        } else {
            $content | Set-Content -Path $DestinationPath -Encoding UTF8
        }

        return $true
    } catch {
        Write-Error "Error downloading ${SourcePath}: $_"
        return $false
    }
}
