function Get-LinesToRedraw {
    <#
    .SYNOPSIS
        Determines which lines need to be redrawn based on changes
    .DESCRIPTION
        Compares current state with previous state to minimize redraws.
        Only returns lines that actually changed (cursor movement, selection, expansion, etc.)
    #>
    param(
        [int]$StartLine,
        [int]$EndLine
    )

    $linesToRedraw = @()

    # If scroll offset changed, redraw everything
    if ($script:previousScrollOffset -ne $script:scrollOffset) {
        for ($i = $StartLine; $i -le $EndLine; $i++) {
            $linesToRedraw += $i
        }
        return $linesToRedraw
    }

    # Check cursor position changes (old and new cursor lines)
    if ($script:previousCursorIndex -ne -1) {
        $prevCursorLine = $StartLine + ($script:previousCursorIndex - $script:scrollOffset)
        if ($prevCursorLine -ge $StartLine -and $prevCursorLine -le $EndLine) {
            $linesToRedraw += $prevCursorLine
        }
    }

    $currentCursorLine = $StartLine + ($script:currentIndex - $script:scrollOffset)
    if ($currentCursorLine -ge $StartLine -and $currentCursorLine -le $EndLine) {
        if ($linesToRedraw -notcontains $currentCursorLine) {
            $linesToRedraw += $currentCursorLine
        }
    }

    # Detect if this is a list structure change (expansion/collapse)
    # by checking if any line has a different node URL than before
    $firstChangedLine = -1
    for ($i = $StartLine; $i -le $EndLine; $i++) {
        $nodeIndex = $script:scrollOffset + ($i - $StartLine)
        if ($nodeIndex -ge 0 -and $nodeIndex -lt $script:flattenedNodes.Count) {
            $currentNode = $script:flattenedNodes[$nodeIndex]

            if ($script:previouslyDrawnNodes.ContainsKey($i)) {
                $prevNodeData = $script:previouslyDrawnNodes[$i]

                # If the node URL changed, the structure changed (expand/collapse)
                if ($prevNodeData.Url -ne $currentNode.Url) {
                    $firstChangedLine = $i
                    break
                }
            }
        } elseif ($script:previouslyDrawnNodes.ContainsKey($i)) {
            # Line now empty but had content - structure changed
            $firstChangedLine = $i
            break
        }
    }

    # If structure changed, redraw from that point downward
    if ($firstChangedLine -ne -1) {
        for ($i = $firstChangedLine; $i -le $EndLine; $i++) {
            if ($linesToRedraw -notcontains $i) {
                $linesToRedraw += $i
            }
        }
        return $linesToRedraw | Sort-Object
    }

    # No structure change - check for state changes (selection, expansion icon)
    for ($i = $StartLine; $i -le $EndLine; $i++) {
        $nodeIndex = $script:scrollOffset + ($i - $StartLine)
        if ($nodeIndex -ge 0 -and $nodeIndex -lt $script:flattenedNodes.Count) {
            $currentNode = $script:flattenedNodes[$nodeIndex]

            # Check if this line had different state before
            if ($script:previouslyDrawnNodes.ContainsKey($i)) {
                $prevNodeData = $script:previouslyDrawnNodes[$i]

                # Compare selection state and expansion state
                if ($prevNodeData.IsSelected -ne $currentNode.IsSelected -or
                    $prevNodeData.IsExpanded -ne $currentNode.IsExpanded) {
                    if ($linesToRedraw -notcontains $i) {
                        $linesToRedraw += $i
                    }
                }
            } else {
                # New node on this line
                if ($linesToRedraw -notcontains $i) {
                    $linesToRedraw += $i
                }
            }
        } elseif ($script:previouslyDrawnNodes.ContainsKey($i)) {
            # Line now empty but had content before - need to clear it
            if ($linesToRedraw -notcontains $i) {
                $linesToRedraw += $i
            }
        }
    }

    return $linesToRedraw | Sort-Object
}

function Clear-UnusedLines {
    <#
    .SYNOPSIS
        Clears lines that no longer contain content
    .DESCRIPTION
        Removes visual artifacts from lines that previously had content but are now empty
    #>
    param(
        [int]$StartLine,
        [int]$EndLine,
        [int]$CurrentContentMaxLine,
        [array]$LinesToRedraw,
        [bool]$IsFullRedraw
    )

    $linesToClear = @()

    # Method 1: Check previously drawn nodes that are now beyond current content
    foreach ($prevLineNum in $script:previouslyDrawnNodes.Keys) {
        if ($prevLineNum -gt $CurrentContentMaxLine -and $prevLineNum -lt $EndLine) {
            if ($linesToClear -notcontains $prevLineNum) {
                $linesToClear += $prevLineNum
            }
        }
    }

    # Method 2: On full redraw, ensure all lines beyond content are cleared
    # This catches any lines that might have been missed (safety net)
    if ($IsFullRedraw) {
        for ($i = ($CurrentContentMaxLine + 1); $i -le $EndLine; $i++) {
            if ($linesToClear -notcontains $i -and $LinesToRedraw -notcontains $i) {
                $linesToClear += $i
            }
        }
    }

    # Clear all identified lines
    foreach ($lineNum in $linesToClear) {
        [Console]::SetCursorPosition(0, $lineNum)
        [Console]::Write(" " * [Console]::WindowWidth)
        $script:previouslyDrawnNodes.Remove($lineNum)
    }
}

function Draw-Tree {
    <#
    .SYNOPSIS
        Renders the tree UI to the console
    .DESCRIPTION
        Main rendering function that draws the tree structure with selective redrawing
        for optimal performance. Uses StringBuilder and direct Console API calls.
    .PARAMETER Root
        The root TreeNode to render
    .PARAMETER FullRedraw
        Forces a complete redraw of the entire screen
    .PARAMETER DrawHeaderFooter
        Whether to draw the header and footer
    #>
    param(
        [TreeNode]$Root,
        [bool]$FullRedraw = $false,
        [bool]$DrawHeaderFooter = $false
    )

    try {
        [Console]::CursorVisible = $false
    } catch {
        # Ignore if console handle is not available
    }

    # Only clear on full redraw, otherwise we'll selectively update
    if ($FullRedraw) {
        try {
            [Console]::Clear()
        } catch {
            Clear-Host
        }
        # Reset tracking state on full clear
        $script:previouslyDrawnNodes = @{}
        $script:previousCursorIndex = -1
        $script:previousScrollOffset = -1
        $DrawHeaderFooter = $true  # Always draw header/footer after clear
    }

    # Cache console dimensions (avoid repeated property access)
    $windowWidth = [Console]::WindowWidth
    $windowHeight = [Console]::WindowHeight - $script:Config_ReservedLines

    try {
        # Reserve space for header and footer (configured lines)
        $maxNodes = [Math]::Min($script:flattenedNodes.Count, $windowHeight)
        $endIndex = [Math]::Min($script:scrollOffset + $maxNodes - 1, $script:flattenedNodes.Count - 1)
        $visibleNodes = $script:flattenedNodes[$script:scrollOffset..$endIndex]

        # Header - only draw when explicitly requested
        if ($DrawHeaderFooter) {
            $header = $script:Config_ApplicationName

            # Calculate exact padding to center the header
            $totalPadding = $windowWidth - $header.Length
            if ($totalPadding -lt 0) {
                # Header too long, truncate it
                $header = $header.Substring(0, $windowWidth)
                $totalPadding = 0
            }

            $leftPadding = [Math]::Floor($totalPadding / 2)
            $rightPadding = $totalPadding - $leftPadding

            [Console]::SetCursorPosition(0, 0)
            $headerLine = (" " * $leftPadding) + $header + (" " * $rightPadding)
            # Ensure exact width
            if ($headerLine.Length -gt $windowWidth) {
                $headerLine = $headerLine.Substring(0, $windowWidth)
            }
            [Console]::Write("$($script:Theme.HeaderBg)$($script:Theme.HeaderFg)$headerLine$($script:Theme.Reset)`n")

            # Separator line
            $separatorLine = "-" * $windowWidth
            if ($separatorLine.Length -gt $windowWidth) {
                $separatorLine = $separatorLine.Substring(0, $windowWidth)
            }
            [Console]::Write("$($script:Theme.SeparatorFg)$separatorLine$($script:Theme.Reset)`n")
        }
    } catch {
        throw "Error drawing header: $_"
    }

    # Tree content
    $startLine = 2
    $endLine = [Math]::Min($startLine + $visibleNodes.Count - 1, $startLine + $windowHeight - 1)

    # Determine which lines need redrawing
    if ($FullRedraw) {
        # Full redraw - draw all lines
        $linesToRedraw = @($startLine..$endLine)
    } else {
        # Selective redraw - only changed lines
        $linesToRedraw = Get-LinesToRedraw -StartLine $startLine -EndLine $endLine
    }

    # Draw only the lines that need updating
    foreach ($lineNum in $linesToRedraw) {
        $i = $lineNum - $startLine

        if ($i -ge 0 -and $i -lt $visibleNodes.Count) {
            $node = $visibleNodes[$i]
            $actualIndex = $script:scrollOffset + $i
            $isCurrentLine = ($actualIndex -eq $script:currentIndex)

            # Determine if this is the last sibling
            $isLast = $false
            if ($node.Parent -ne $null) {
                $siblings = $node.Parent.Children
                $isLast = ($siblings[-1] -eq $node)
            }

            $prefix = Get-TreePrefix -Node $node -IsLast $isLast

            # Node symbol and color
            $symbol = ""
            $symbolColor = $script:Theme.TreeStructureFg
            $isEmpty = $false

            if ($node.IsDirectory) {
                if ($node.Children.Count -eq 0 -and $node.IsExpanded) {
                    # Empty directory (after expansion)
                    $symbol = $script:symbols.FolderEmpty
                    $symbolColor = $script:Theme.EmptyFolderSymbolFg
                    $isEmpty = $true
                } elseif ($node.IsExpanded) {
                    $symbol = $script:symbols.FolderOpen
                } else {
                    $symbol = $script:symbols.FolderClosed
                }
            } else {
                $symbol = $script:symbols.File
            }

            # Checkbox
            $checkbox = if ($node.IsSelected) { $script:symbols.CheckboxFilled } else { $script:symbols.CheckboxEmpty }

            # Determine colors
            $fgColor = $script:Theme.DefaultFileFg
            if ($node.IsSelected) {
                $fgColor = $script:Theme.SelectedFg
            } elseif ($node.IsDirectory -and -not $isEmpty) {
                $fgColor = $script:Theme.DirectoryFg
            } elseif ($node.IsDirectory -and $isEmpty) {
                $fgColor = $script:Theme.EmptyFolderTextFg
            } elseif ($node.Name -match '\.(ps1|bat|cmd)$') {
                $fgColor = $script:Theme.ScriptFg
            }

            # Build entire line as single string (major performance improvement)
            [Console]::SetCursorPosition(0, $lineNum)

            $lineBuilder = [System.Text.StringBuilder]::new(256)

            # Cursor indicator
            if ($isCurrentLine) {
                [void]$lineBuilder.Append("$($script:Theme.CurrentLineBg)$($script:Theme.CurrentLineFg)> $($script:Theme.Reset)")
            } else {
                [void]$lineBuilder.Append("  ")
            }

            # Tree structure, checkbox, symbol, and name - all in one string
            [void]$lineBuilder.Append("$($script:Theme.TreeStructureFg)$prefix$checkbox $($script:Theme.Reset)")
            [void]$lineBuilder.Append("$symbolColor$symbol $($script:Theme.Reset)")

            # Pad the name to clear old content on this line
            $remainingWidth = $windowWidth - 2 - $prefix.Length - $checkbox.Length - 1 - $symbol.Length - 1
            $paddedName = $node.Name.PadRight($remainingWidth)
            if ($paddedName.Length -gt $remainingWidth) {
                $paddedName = $paddedName.Substring(0, $remainingWidth)
            }
            [void]$lineBuilder.Append("$fgColor$paddedName$($script:Theme.Reset)")

            # Write entire line in one call
            [Console]::Write($lineBuilder.ToString())

            # Update tracking state
            $script:previouslyDrawnNodes[$lineNum] = @{
                Url = $node.Url
                IsSelected = $node.IsSelected
                IsExpanded = $node.IsExpanded
            }
        } else {
            # Clear line that's beyond visible nodes
            [Console]::SetCursorPosition(0, $lineNum)
            [Console]::Write(" " * $windowWidth)
            $script:previouslyDrawnNodes.Remove($lineNum)
        }
    }

    # Clear any lines that are no longer needed (consolidated clearing logic)
    $currentMaxLine = $startLine + $visibleNodes.Count - 1
    Clear-UnusedLines -StartLine $startLine -EndLine ($startLine + $windowHeight - 1) `
                      -CurrentContentMaxLine $currentMaxLine -LinesToRedraw $linesToRedraw `
                      -IsFullRedraw $FullRedraw

    # Update tracking state
    $script:previousCursorIndex = $script:currentIndex
    $script:previousScrollOffset = $script:scrollOffset

    # Footer - only draw when explicitly requested
    if ($DrawHeaderFooter) {
        [Console]::SetCursorPosition(0, [Console]::WindowHeight - 1)
        $footer = "Navigate: Arrows | Enter: Expand | Space: Select | P: Presets | R: Run | Esc: Exit"

        # Calculate exact padding to center the footer (use cached windowWidth)
        $totalPadding = $windowWidth - $footer.Length
        if ($totalPadding -lt 0) {
            # Footer too long, truncate it
            $footer = $footer.Substring(0, $windowWidth)
            $totalPadding = 0
        }

        $leftPadding = [Math]::Floor($totalPadding / 2)
        $rightPadding = $totalPadding - $leftPadding

        $footerLine = (" " * $leftPadding) + $footer + (" " * $rightPadding)
        # Ensure exact width
        if ($footerLine.Length -gt $windowWidth) {
            $footerLine = $footerLine.Substring(0, $windowWidth)
        }

        [Console]::Write("$($script:Theme.FooterBg)$($script:Theme.FooterFg)$footerLine$($script:Theme.Reset)")
    }
}
