function Initialize-DefaultTheme {
    <#
    .SYNOPSIS
        Initializes a default dark theme for the tree UI
    .DESCRIPTION
        Creates a simple, static dark theme with ANSI escape codes.
        Consumers can override this by setting $script:Theme before calling this function.
    .OUTPUTS
        Hashtable with theme colors (ANSI escape codes)
    #>

    # ANSI escape code helper
    $esc = [char]27

    # Helper function to convert RGB to ANSI foreground code
    function Get-AnsiRgbFg($r, $g, $b) {
        return "$esc[38;2;$r;$g;${b}m"
    }

    # Helper function to convert RGB to ANSI background code
    function Get-AnsiRgbBg($r, $g, $b) {
        return "$esc[48;2;$r;$g;${b}m"
    }

    # Default Dark Theme
    $theme = @{
        Reset = "$esc[0m"

        # Header and Footer
        HeaderBg = Get-AnsiRgbBg 0 0 139        # DarkBlue
        HeaderFg = Get-AnsiRgbFg 255 255 255    # White
        FooterBg = Get-AnsiRgbBg 0 0 139        # DarkBlue
        FooterFg = Get-AnsiRgbFg 255 255 255    # White

        # Current line (selected/cursor position)
        CurrentLineBg = Get-AnsiRgbBg 0 0 139   # DarkBlue
        CurrentLineFg = Get-AnsiRgbFg 255 255 255 # White

        # Tree structure elements
        SeparatorFg = Get-AnsiRgbFg 128 128 128       # Gray
        TreeStructureFg = Get-AnsiRgbFg 128 128 128   # Gray

        # File and folder colors
        DirectoryFg = Get-AnsiRgbFg 255 255 0         # Yellow
        SelectedFg = Get-AnsiRgbFg 0 255 0            # Green
        ScriptFg = Get-AnsiRgbFg 0 255 255            # Cyan
        EmptyFolderSymbolFg = Get-AnsiRgbFg 255 0 0   # Red
        EmptyFolderTextFg = Get-AnsiRgbFg 169 169 169 # DarkGray
        DefaultFileFg = Get-AnsiRgbFg 128 128 128     # Gray

        # Status colors
        ErrorFg = Get-AnsiRgbFg 255 0 0               # Red
        WarningFg = Get-AnsiRgbFg 255 255 0           # Yellow
        SuccessFg = Get-AnsiRgbFg 0 255 0             # Green
        InfoFg = Get-AnsiRgbFg 128 128 128            # Gray

        # Popup window colors
        PopupBg = Get-AnsiRgbBg 0 0 0                 # Black
        PopupTitleFg = Get-AnsiRgbFg 255 255 0        # Yellow
        PopupTextFg = Get-AnsiRgbFg 255 255 255       # White
        PopupHighlightFg = Get-AnsiRgbFg 0 255 255    # Cyan
        PopupMutedFg = Get-AnsiRgbFg 169 169 169      # DarkGray
        PopupWarningFg = Get-AnsiRgbFg 184 134 11     # DarkYellow
        PopupYesFg = Get-AnsiRgbFg 0 255 0            # Green
        PopupNoFg = Get-AnsiRgbFg 255 0 0             # Red
    }

    return $theme
}

function Initialize-DefaultSymbols {
    <#
    .SYNOPSIS
        Initializes default tree symbols
    .DESCRIPTION
        Creates a hashtable with default ASCII tree symbols.
        Consumers can override this by setting $script:symbols before calling.
    .OUTPUTS
        Hashtable with tree symbols
    #>

    return @{
        FolderClosed = "[+]"
        FolderOpen = "[-]"
        FolderEmpty = "[!]"
        File = " > "
        Branch = "|--"
        LastBranch = "+--"
        Vertical = "|  "
        Space = "   "
        CheckboxEmpty = "[ ]"
        CheckboxFilled = "[X]"
    }
}
