#Requires -Version 5.1

<#
.SYNOPSIS
    ITM.TreeUI - Interactive Tree UI Module
.DESCRIPTION
    Provides a reusable tree-based UI component for PowerShell console applications.
    Features selective redrawing, keyboard navigation, and theme support.
.NOTES
    Version: 1.0.0
    Author: Farm Town Tech
.EXAMPLE
    # Basic usage with default theme
    Import-Module ITM.TreeUI
    $root = [TreeNode]::new("Root", "http://example.com", $true, $null, 0)
    $root.IsExpanded = $true
    Update-FlattenedList -Root $root
    Draw-Tree -Root $root -FullRedraw $true -DrawHeaderFooter $true

.EXAMPLE
    # Custom theme
    Import-Module ITM.TreeUI
    $script:Theme = @{
        HeaderBg = "`e[48;2;255;0;0m"  # Red background
        # ... other colors ...
    }
    # Use TreeUI as normal
#>

# Module-level script variables for state tracking
$script:currentIndex = 0
$script:flattenedNodes = @()
$script:scrollOffset = 0
$script:previouslyDrawnNodes = @{}
$script:previousCursorIndex = -1
$script:previousScrollOffset = -1

# Import class definition
. "$PSScriptRoot\Classes\TreeNode.ps1"

# Import functions
. "$PSScriptRoot\Functions\Get-TreePrefix.ps1"
. "$PSScriptRoot\Functions\Update-FlattenedList.ps1"
. "$PSScriptRoot\Functions\Draw-Tree.ps1"
. "$PSScriptRoot\Functions\Initialize-Theme.ps1"

# Initialize default theme and symbols if not already set
if (-not $script:Theme) {
    $script:Theme = Initialize-DefaultTheme
}
if (-not $script:symbols) {
    $script:symbols = Initialize-DefaultSymbols
}

# Export public functions and classes
Export-ModuleMember -Function @(
    'Draw-Tree'
    'Update-FlattenedList'
    'Get-TreePrefix'
    'Initialize-DefaultTheme'
    'Initialize-DefaultSymbols'
) -Variable @(
    'currentIndex'
    'flattenedNodes'
    'scrollOffset'
    'Theme'
    'symbols'
)
